<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\TruckMaintenance;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;

class SendMaintenanceNotifications extends Command
{
    protected $signature = 'notifications:maintenance';
    protected $description = 'Kirim notifikasi pengingat maintenance ke aplikasi mobile';

    public function handle()
    {
        // Cek maintenance yang akan datang dalam 7 hari dan belum dinotifikasi
        $maintenances = TruckMaintenance::whereNotNull('next_maintenance_date')
            ->where('notification_sent', false)
            ->whereDate('next_maintenance_date', '<=', Carbon::now()->addDays(7))
            ->whereDate('next_maintenance_date', '>=', Carbon::now())
            ->with('truck')
            ->get();

        foreach ($maintenances as $maintenance) {
            $daysLeft = Carbon::now()->diffInDays($maintenance->next_maintenance_date);
            
            try {
                // Kirim notifikasi ke OneSignal
                $response = // Ganti bagian HTTP request dengan:
                Http::withOptions([
                    'verify' => false
                ])->withHeaders([
                    'Authorization' => 'Basic ' . env('ONESIGNAL_REST_API_KEY'),
                    'accept' => 'application/json',
                    'content-type' => 'application/json'
                ])->post('https://onesignal.com/api/v1/notifications', [
                    'app_id' => env('ONESIGNAL_APP_ID'),
                    'included_segments' => ['Subscribed Users'],
                    'contents' => [
                        'en' => "Maintenance {$maintenance->maintenance_type_label} untuk {$maintenance->truck->license_plate} dalam {$daysLeft} hari"
                    ],
                    'headings' => [
                        'en' => 'Pengingat Maintenance Armada'
                    ],
                    'data' => [
                        'maintenance_id' => $maintenance->id,
                        'truck_id' => $maintenance->truck_id,
                        'type' => 'maintenance_reminder'
                    ]
                ]);

                if ($response->successful()) {
                    // Update status notifikasi
                    $maintenance->update([
                        'notification_sent' => true,
                        'last_notification_sent_at' => now()
                    ]);

                    $this->info("Notifikasi terkirim untuk maintenance {$maintenance->id}");
                } else {
                    $this->error("Gagal mengirim notifikasi: " . $response->body());
                }
            } catch (\Exception $e) {
                $this->error("Error saat mengirim notifikasi: {$e->getMessage()}");
            }
        }

        return 0;
    }
}