<?php

namespace App\Http\Controllers;

use App\Models\Driver;
use App\Models\Truck;
use App\Models\Trip;
use App\Models\FuelPurchase;
use App\Models\Project;
use App\Models\Income;
use App\Models\Expense;
use App\Models\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        // Jumlah driver, truck, dan trip
        $driverCount = Driver::where('status', 'aktif')->count();
        $truckCount = Truck::where('status', 'active')->count();
        $activeTrips = Trip::where('status', 'ongoing')->count();
        
        // Total jarak tempuh bulan ini
        $currentMonthDistance = Trip::whereMonth('start_date', now()->month)
            ->whereYear('start_date', now()->year)
            ->sum('distance');
            
        // Total konsumsi bahan bakar bulan ini
        $currentMonthFuel = FuelPurchase::whereMonth('purchase_date', now()->month)
            ->whereYear('purchase_date', now()->year)
            ->sum('total_cost');
            
        // Trip terbaru
        $recentTrips = Trip::with(['driver', 'truck', 'project'])
            ->latest()
            ->take(5)
            ->get();
            
        // Truck yang perlu service
        $trucksNeedService = Truck::whereDate('next_service_date', '<=', now()->addDays(7))
            ->get();
            
        // Truck dengan STNK yang akan expired atau sudah expired
        $trucksWithStnkIssues = Truck::whereNotNull('stnk_expire_date')
            ->whereDate('stnk_expire_date', '<=', now()->addDays(30))
            ->get();
            
        // Truck dengan KIR yang akan expired atau sudah expired
        $trucksWithKirIssues = Truck::whereNotNull('kir_expire_date')
            ->whereDate('kir_expire_date', '<=', now()->addDays(30))
            ->get();
            
        // Grafik perjalanan per bulan (6 bulan terakhir)
        $tripStats = Trip::select(DB::raw('MONTH(start_date) as month'), DB::raw('COUNT(*) as count'))
            ->whereYear('start_date', now()->year)
            ->whereMonth('start_date', '>=', now()->subMonths(6)->month)
            ->groupBy(DB::raw('MONTH(start_date)'))
            ->get();
            
        // Tambahkan data keuangan
        $currentMonthIncome = Income::whereMonth('income_date', now()->month)
            ->whereYear('income_date', now()->year)
            ->where('approval_status', 'approved')
            ->sum('amount');
            
        $currentMonthExpense = Expense::whereMonth('expense_date', now()->month)
            ->whereYear('expense_date', now()->year)
            ->where('status', 'approved')
            ->sum('amount');
            
        $previousMonthIncome = Income::whereMonth('income_date', now()->subMonth()->month)
            ->whereYear('income_date', now()->subMonth()->year)
            ->where('approval_status', 'approved')
            ->sum('amount');
            
        $previousMonthExpense = Expense::whereMonth('expense_date', now()->subMonth()->month)
            ->whereYear('expense_date', now()->subMonth()->year)
            ->where('status', 'approved')
            ->sum('amount');
        
        // Proyek dengan deadline terdekat
        $upcomingProjects = Project::where('status', 'active')
            ->where('end_date', '>=', now())
            ->orderBy('end_date', 'asc')
            ->take(5)
            ->get();
        
        // Top clients by project value
        $topClients = Client::withCount('projects')
            ->withSum('projects as total_budget', 'budget')
            ->orderBy('total_budget', 'desc')
            ->take(5)
            ->get();
            
        // Pemakaian BBM teratas (bulan ini)
        $topFuelConsumption = Truck::select('trucks.id', 'trucks.plate_number', 'trucks.brand', 'trucks.model')
            ->addSelect(DB::raw('SUM(fuel_purchases.quantity) as total_fuel'))
            ->join('fuel_purchases', 'trucks.id', '=', 'fuel_purchases.truck_id')
            ->whereMonth('fuel_purchases.purchase_date', now()->month)
            ->whereYear('fuel_purchases.purchase_date', now()->year)
            ->groupBy('trucks.id', 'trucks.plate_number', 'trucks.brand', 'trucks.model')
            ->orderBy('total_fuel', 'desc')
            ->take(5)
            ->get();
            
        // Retase teratas (bulan ini)
        $topTrips = Truck::select('trucks.id', 'trucks.plate_number', 'trucks.brand', 'trucks.model')
            ->addSelect(DB::raw('COUNT(trips.id) as trip_count'))
            ->addSelect(DB::raw('SUM(trips.distance) as total_distance'))
            ->join('trips', 'trucks.id', '=', 'trips.truck_id')
            ->whereMonth('trips.start_date', now()->month)
            ->whereYear('trips.start_date', now()->year)
            ->groupBy('trucks.id', 'trucks.plate_number', 'trucks.brand', 'trucks.model')
            ->orderBy('trip_count', 'desc')
            ->take(5)
            ->get();
        
        return view('dashboard', compact(
            'driverCount', 
            'truckCount', 
            'activeTrips',
            'currentMonthDistance', 
            'currentMonthFuel',
            'recentTrips',
            'trucksNeedService',
            'trucksWithStnkIssues',
            'trucksWithKirIssues',
            'tripStats',
            'currentMonthIncome',
            'currentMonthExpense',
            'previousMonthIncome',
            'previousMonthExpense',
            'upcomingProjects',
            'topClients',
            'topFuelConsumption',
            'topTrips'
        ));
    }
}
