<?php

namespace App\Http\Controllers;

use App\Models\DriverActivity;
use App\Models\Driver;
use App\Models\Truck;
use App\Models\ManualTripCount;
use App\Models\FuelPurchase;
use App\Models\Trip;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class DriverActivityController extends Controller
{
    public function store(Request $request): JsonResponse
    {
        \Log::info('DEBUG material_np_id dari request:', ['material_np_id' => $request->input('material_np_id'), 'all' => $request->all()]);
        $validated = $request->validate([
            'hariTanggal' => 'required|date',
            'platNomor' => 'required|string',
            'asal' => 'required|string',
            'tujuan' => 'required|string',
            'kmStart' => 'required|integer',
            'kmEnd' => 'required|integer',
            'pengisianSolar' => 'required|integer',
            'indikatorSolar' => 'required|integer',
            'retase' => 'required|integer',
            'material_np_id' => 'nullable|exists:material_nps,id'
        ]);

        try {
            DB::beginTransaction();
            
            // 1. Simpan aktivitas driver
            $activity = DriverActivity::create([
                'hari_tanggal' => $validated['hariTanggal'],
                'plat_nomor' => $validated['platNomor'],
                'asal' => $validated['asal'],
                'tujuan' => $validated['tujuan'],
                'km_start' => $validated['kmStart'],
                'km_end' => $validated['kmEnd'],
                'pengisian_solar' => $validated['pengisianSolar'],
                'indikator_solar' => $validated['indikatorSolar'],
                'retase' => $validated['retase'],
                'user_id' => auth()->id(),
                'material_np_id' => $validated['material_np_id'] ?? null
            ]);
            
            // 2. Cari truck berdasarkan plat nomor
            $truck = Truck::where('plate_number', $validated['platNomor'])->first();
            
            if ($truck) {
                // 3. Cari driver yang menggunakan truck tersebut
                $driver = Driver::where('truck_id', $truck->id)->first();
                
                if ($driver) {
                    // 4. Update atau buat ManualTripCount untuk driver
                    // Selalu insert baru per tanggal dari aplikasi
                    // Ubah format tanggal ke Y-m-d
                    try {
                        $date = \Carbon\Carbon::createFromFormat('d M Y', $validated['hariTanggal'])->format('Y-m-d');
                    } catch (\Exception $e) {
                        \Log::error('Format tanggal tidak valid untuk ManualTripCount', ['input' => $validated['hariTanggal']]);
                        throw $e;
                    }
                    \Log::info('Insert ManualTripCount', [
                        'driver_id' => $driver->id,
                        'date' => $date,
                        'completed_trips' => $validated['retase'],
                        'ongoing_trips' => 0,
                        'notes' => 'Dibuat otomatis dari aplikasi driver pada ' . now()->format('d/m/Y H:i'),
                        'added_by' => auth()->id()
                    ]);
                    ManualTripCount::create([
                        'driver_id' => $driver->id,
                        'date' => $date,
                        'completed_trips' => $validated['retase'],
                        'ongoing_trips' => 0,
                        'notes' => 'Dibuat otomatis dari aplikasi driver pada ' . now()->format('d/m/Y H:i'),
                        'added_by' => auth()->id()
                    ]);
                    
                    // 5. Jika ada pengisian solar, catat sebagai FuelPurchase
                    if ($validated['pengisianSolar'] > 0) {
                        FuelPurchase::create([
                            'truck_id' => $truck->id,
                            'purchase_date' => $validated['hariTanggal'],
                            'quantity' => $validated['pengisianSolar'],
                            'price_per_liter' => 0, // Harga per liter bisa diupdate manual nanti
                            'total_cost' => 0, // Total cost bisa diupdate manual nanti
                            'fuel_type' => $truck->fuel_type ?? 'Solar',
                            'station_name' => 'Diisi via aplikasi driver',
                            'driver_name' => $driver->name,
                            'notes' => 'Dicatat otomatis dari aplikasi driver pada ' . now()->format('d/m/Y H:i')
                        ]);
                    }
                    
                    // 6. Catat perjalanan ke tabel trips
                    $trip = Trip::create([
                        'driver_id' => $driver->id,
                        'truck_id' => $truck->id,
                        'project_id' => null, // Bisa diupdate manual nanti
                        'material_np_id' => $validated['material_np_id'] ?? null, // Tambahkan material_np_id
                        'origin' => $validated['asal'],
                        'destination' => $validated['tujuan'],
                        'start_date' => $validated['hariTanggal'],
                        'end_date' => $validated['hariTanggal'], // Karena data diinput setelah perjalanan selesai
                        'start_odometer' => $validated['kmStart'],
                        'end_odometer' => $validated['kmEnd'],
                        'distance' => $validated['kmEnd'] - $validated['kmStart'],
                        'fuel_consumption' => $validated['pengisianSolar'] > 0 ? $validated['pengisianSolar'] : null,
                        'indikator_solar' => $validated['indikatorSolar'], // Tambahkan indikator solar
                        'status' => 'completed', // Karena data diinput setelah perjalanan selesai
                        'notes' => 'Dicatat otomatis dari aplikasi driver pada ' . now()->format('d/m/Y H:i')
                    ]);
                    
                    // Update DriverActivity dengan trip_id
                    $activity->update([
                        'trip_id' => $trip->id
                    ]);
                }
            }
            
            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Data aktivitas berhasil disimpan dan didistribusikan',
                'data' => $activity
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'status' => 'error',
                'message' => 'Gagal menyimpan data aktivitas: ' . $e->getMessage()
            ], 500);
        }
    }

    public function index(Request $request): JsonResponse
    {
        try {
            // Ambil aktivitas driver untuk user yang sedang login
            $activities = DriverActivity::where('user_id', auth()->id())
            ->with('materialNp')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($activity) {
                    return [
                        'tanggal' => $activity->hari_tanggal,
                        'plat_nomor' => $activity->plat_nomor,
                        'asal' => $activity->asal,
                        'tujuan' => $activity->tujuan,
                        'km_start' => $activity->km_start,
                        'km_end' => $activity->km_end,
                        'pengisian_solar' => $activity->pengisian_solar,
                        'indikator_solar' => $activity->indikator_solar,
                        'retase' => $activity->retase,
                        'material_np_id' => $activity->material_np_id,
                        'material_np_name' => optional($activity->materialNp)->name,
                        'material_np_unit' => optional($activity->materialNp)->unit
                    ];
                });
    
            return response()->json([
                'status' => 'success',
                'data' => $activities
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Gagal mengambil data aktivitas: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getDriverActivitiesGrouped(Request $request)
    {
        // Ambil aktivitas driver untuk user tertentu atau semua jika admin
        $query = DriverActivity::query();
        
        if (!auth()->user()->hasRole('admin')) {
            $query->where('user_id', auth()->id());
        }
        
        // Jika ada parameter driver_id, filter berdasarkan driver
        if ($request->has('driver_id')) {
            $driver = Driver::find($request->driver_id);
            if ($driver && $driver->truck) {
                $query->where('plat_nomor', $driver->truck->plate_number);
            }
        }
        
        // Jika ada parameter truck_id, filter berdasarkan truck
        if ($request->has('truck_id')) {
            $truck = Truck::find($request->truck_id);
            if ($truck) {
                $query->where('plat_nomor', $truck->plate_number);
            }
        }
        
        // Ambil data dan kelompokkan berdasarkan tanggal
        $activities = $query->orderBy('hari_tanggal', 'desc')
                            ->get()
                            ->groupBy(function($activity) {
                                return $activity->hari_tanggal->format('Y-m-d');
                            });
        
        return view('driver-activities.index', compact('activities'));
    }
}