<?php

namespace App\Http\Controllers;

use App\Models\Driver;
use App\Models\Truck;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class DriverController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $drivers = Driver::with('trips')->latest()->paginate(10);
        return view('drivers.index', compact('drivers'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $trucks = Truck::where('status', 'active')->get();
        return view('drivers.create', compact('trucks'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'id_number' => 'required|string|max:255|unique:drivers',
            'license_number' => 'required|string|max:255|unique:drivers',
            'license_expiry' => 'required|date',
            'phone' => 'required|string|max:255',
            'address' => 'required|string',
            'status' => 'required|string|in:aktif,cuti,tidak aktif',
            'join_date' => 'required|date',
            'exit_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'emergency_contact' => 'nullable|string|max:255',
            'blood_type' => 'nullable|string|max:10',
            'profile_photo' => 'nullable|image|max:2048',
            'salary' => 'nullable|numeric|min:0',
            'truck_id' => 'nullable|exists:trucks,id',
        ]);

        if ($request->hasFile('profile_photo')) {
            $path = $request->file('profile_photo')->store('drivers', 'public');
            $validated['profile_photo'] = $path;
        }

        // Generate email dan username dari nama driver
        $email = Str::slug($validated['name'], '.') . '@obm.com';
        $username = Str::slug($validated['name'], '_');
        $counter = 1;

        // Pastikan email dan username unik
        while (User::where('email', $email)->exists()) {
            $email = Str::slug($validated['name'], '.') . $counter . '@obm.com';
            $counter++;
        }

        $counter = 1;
        while (User::where('username', $username)->exists()) {
            $username = Str::slug($validated['name'], '_') . '_' . $counter;
            $counter++;
        }

        // Tambahkan email ke data driver
        $validated['email'] = $email;

        // Buat driver baru
        $driver = Driver::create($validated);

        // Buat user baru
        $user = User::create([
            'name' => $validated['name'],
            'email' => $email,
            'username' => $username,
            'password' => Hash::make('password'), // Password default
            'email_verified_at' => now(),
            'remember_token' => Str::random(10),
        ]);

        // Berikan role driver
        $user->assignRole('driver');

        return redirect()->route('drivers.index')
            ->with('success', 'Driver berhasil ditambahkan dengan username: ' . $username);
    }

    /**
     * Display the specified resource.
     */
    public function show(Driver $driver)
    {
        $driver->load('trips', 'truck');
        return view('drivers.show', compact('driver'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Driver $driver)
    {
        $trucks = Truck::where('status', 'active')->get();
        return view('drivers.edit', compact('driver', 'trucks'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Driver $driver)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'id_number' => 'required|string|max:255|unique:drivers,id_number,' . $driver->id,
            'license_number' => 'required|string|max:255|unique:drivers,license_number,' . $driver->id,
            'license_expiry' => 'required|date',
            'phone' => 'required|string|max:255',
            'address' => 'required|string',
            'status' => 'required|string|in:aktif,cuti,tidak aktif',
            'join_date' => 'required|date',
            'exit_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'emergency_contact' => 'nullable|string|max:255',
            'blood_type' => 'nullable|string|max:10',
            'profile_photo' => 'nullable|image|max:2048',
            'salary' => 'nullable|numeric|min:0',
            'truck_id' => 'nullable|exists:trucks,id',
            'new_password' => 'nullable|string|min:8|confirmed',
        ]);
    
        if ($request->hasFile('profile_photo')) {
            // Hapus foto lama jika ada
            if ($driver->profile_photo) {
                Storage::disk('public')->delete($driver->profile_photo);
            }
            $path = $request->file('profile_photo')->store('drivers', 'public');
            $validated['profile_photo'] = $path;
        }
    
        $driver->update($validated);
    
        // Update password jika ada
        if ($request->filled('new_password') && $driver->user) {
            $driver->user->update([
                'password' => Hash::make($request->new_password)
            ]);
        }
    
        return redirect()->route('drivers.index')
            ->with('success', 'Data driver berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Driver $driver)
    {
        // Hapus foto jika ada
        if ($driver->profile_photo) {
            Storage::disk('public')->delete($driver->profile_photo);
        }

        $driver->delete();

        return redirect()->route('drivers.index')
            ->with('success', 'Driver berhasil dihapus.');
    }
}
