<?php

namespace App\Http\Controllers;

use App\Models\Expense;
use App\Models\Project;
use App\Exports\ExpensesExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Rap2hpoutre\FastExcel\FastExcel;

class ExpenseController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:manage-finances');
    }
    
    /**
     * Display a listing of expenses.
     */
    public function index(Request $request)
    {
        $query = Expense::with(['project', 'addedBy']);
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('category', 'like', "%{$searchTerm}%");
            });
        }
        
        // Filter berdasarkan tanggal mulai
        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }
        
        // Filter berdasarkan tanggal akhir
        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }
        
        $expenses = $query->latest()->paginate(10)->withQueryString();
        return view('expenses.index', compact('expenses'));
    }

    /**
     * Show the form for creating a new expense.
     */
    public function create()
    {
        $projects = Project::all();
        $categories = ['Bahan Bakar', 'Perawatan', 'Gaji', 'Administrasi', 'Lainnya'];
        $paymentMethods = ['Tunai', 'Transfer Bank', 'Kartu Kredit', 'Lainnya'];
        
        return view('expenses.create', compact('projects', 'categories', 'paymentMethods'));
    }

    /**
     * Store a newly created expense in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'amount' => 'required|numeric|min:0',
            'expense_date' => 'required|date',
            'project_id' => 'nullable|exists:projects,id',
            'category' => 'required|string|max:255',
            'payment_method' => 'required|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
        ]);
        
        $validated['added_by'] = Auth::id();
        
        Expense::create($validated);
        
        return redirect()->route('expenses.index')
            ->with('success', 'Pengeluaran berhasil ditambahkan.');
    }

    /**
     * Display the specified expense.
     */
    public function show(Expense $expense)
    {
        return view('expenses.show', compact('expense'));
    }

    /**
     * Show the form for editing the specified expense.
     */
    public function edit(Expense $expense)
    {
        $projects = Project::all();
        $categories = ['Bahan Bakar', 'Perawatan', 'Gaji', 'Administrasi', 'Lainnya'];
        $paymentMethods = ['Tunai', 'Transfer Bank', 'Kartu Kredit', 'Lainnya'];
        
        return view('expenses.edit', compact('expense', 'projects', 'categories', 'paymentMethods'));
    }

    /**
     * Update the specified expense in storage.
     */
    public function update(Request $request, Expense $expense)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'amount' => 'required|numeric|min:0',
            'expense_date' => 'required|date',
            'project_id' => 'nullable|exists:projects,id',
            'category' => 'required|string|max:255',
            'payment_method' => 'required|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
        ]);
        
        $expense->update($validated);
        
        return redirect()->route('expenses.index')
            ->with('success', 'Pengeluaran berhasil diperbarui.');
    }

    /**
     * Remove the specified expense from storage.
     */
    public function destroy(Expense $expense)
    {
        $expense->delete();
        
        return redirect()->route('expenses.index')
            ->with('success', 'Pengeluaran berhasil dihapus.');
    }
    
    /**
     * Approve the specified expense.
     */
    public function approve(Expense $expense)
    {
        if (!Auth::user()->hasPermission('approve-finances')) {
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Anda tidak memiliki izin untuk menyetujui pengeluaran.'
                ]);
            }
            return redirect()->route('expenses.index')->with('error', 'Anda tidak memiliki izin untuk menyetujui pengeluaran.');
        }

        $expense->update([
            'status' => 'approved',
            'approved_by' => Auth::id(),
            'approved_at' => now()
        ]);

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Pengeluaran berhasil disetujui'
            ]);
        }
        return redirect()->route('expenses.index')->with('success', 'Pengeluaran berhasil disetujui.');
    }

    public function reject(Expense $expense)
    {
        if (!Auth::user()->hasPermission('approve-finances')) {
            return response()->json([
                'success' => false,
                'message' => 'Anda tidak memiliki izin untuk menolak pengeluaran.'
            ]);
        }
        
        $expense->update([
            'status' => 'rejected',
            'approved_by' => Auth::id(),
            'approved_at' => now()
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Pengeluaran berhasil ditolak'
        ]);
    }
    
    /**
     * Export expenses to Excel
     */
    public function export() 
    {
        $exporter = new ExpensesExport();
        return (new FastExcel($exporter->collection()))->download('pengeluaran.xlsx');
    }
}