<?php

namespace App\Http\Controllers;

use App\Models\FuelPurchase;
use App\Models\Expense;
use App\Models\Truck;
use App\Models\Driver;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class FuelPurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $fuelPurchases = FuelPurchase::with('truck')->latest()->paginate(10);
        return view('fuel-purchases.index', compact('fuelPurchases'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $trucks = Truck::where('status', 'active')->get();
        $drivers = Driver::orderBy('name')->get();
        return view('fuel-purchases.create', compact('trucks', 'drivers'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'truck_id' => 'required|exists:trucks,id',
            'purchase_date' => 'required|date',
            'quantity' => 'required|numeric|min:0',
            'price_per_liter' => 'required|numeric|min:0',
            'total_cost' => 'required|numeric|min:0',
            'fuel_type' => 'required|string|max:255',
            'station_name' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'driver_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'receipt_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
        
        $data = $request->all();
        
        // Handle receipt image upload
        if ($request->hasFile('receipt_image')) {
            $path = $request->file('receipt_image')->store('receipts', 'public');
            $data['receipt_image'] = $path;
        }
        
        FuelPurchase::create($data);
        
        return redirect()->route('fuel-purchases.index')
            ->with('success', 'Pengisian bahan bakar berhasil ditambahkan.');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FuelPurchase $fuelPurchase)
    {
        $validated = $request->validate([
            'truck_id' => 'required|exists:trucks,id',
            'purchase_date' => 'required|date',
            'quantity' => 'required|numeric|min:0',
            'price_per_liter' => 'required|numeric|min:0',
            'total_cost' => 'required|numeric|min:0',
            'fuel_type' => 'required|string|max:255',
            'station_name' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'driver_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'receipt_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
        
        $data = $request->all();
        
        // Handle receipt image upload
        if ($request->hasFile('receipt_image')) {
            // Delete old image if exists
            if ($fuelPurchase->receipt_image) {
                Storage::disk('public')->delete($fuelPurchase->receipt_image);
            }
            
            $path = $request->file('receipt_image')->store('receipts', 'public');
            $data['receipt_image'] = $path;
        }
        
        $fuelPurchase->update($data);
        
        return redirect()->route('fuel-purchases.index')
            ->with('success', 'Pengisian bahan bakar berhasil diperbarui.');
    }

    /**
     * Display the specified resource.
     */
    public function show(FuelPurchase $fuelPurchase)
    {
        return view('fuel-purchases.show', compact('fuelPurchase'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(FuelPurchase $fuelPurchase)
    {
        $trucks = Truck::where('status', 'active')->get();
        $drivers = Driver::orderBy('name')->get();
        return view('fuel-purchases.edit', compact('fuelPurchase', 'trucks', 'drivers'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FuelPurchase $fuelPurchase)
    {
        $fuelPurchase->delete();

        return redirect()->route('fuel-purchases.index')
            ->with('success', 'Data pembelian bahan bakar berhasil dihapus.');
    }
}
