<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Driver extends Model
{
    use HasFactory;
    protected $fillable = [
        'name', 'id_number', 'license_number', 'license_expiry',
        'phone', 'address', 'status', 'join_date', 'exit_date',
        'notes', 'emergency_contact', 'blood_type', 'profile_photo',
        'salary', 'truck_id', 'email', 'user_id' // Tambahkan email dan user_id
    ];

    protected $casts = [
        'license_expiry' => 'date',
        'join_date' => 'date',
        'exit_date' => 'date',
    ];

    /**
     * Get the trips associated with the driver.
     */
    public function trips(): HasMany
    {
        return $this->hasMany(Trip::class);
    }

    /**
     * Get the truck assigned to the driver.
     */
    public function truck(): BelongsTo
    {
        return $this->belongsTo(Truck::class);
    }
    
    /**
     * Get the total number of trips (retase) for this driver.
     */
    public function tripCount()
    {
        return $this->trips()->count();
    }
    
    /**
     * Get the total number of completed trips for this driver.
     */
    public function completedTripCount()
    {
        return $this->trips()->where('status', 'completed')->count();
    }
    
    /**
     * Get the total number of ongoing trips for this driver.
     */
    public function ongoingTripCount()
    {
        return $this->trips()->where('status', 'ongoing')->count();
    }

    /**
     * Get the manual trip counts for this driver.
     */
    public function manualTripCounts(): HasMany
    {
        return $this->hasMany(ManualTripCount::class);
    }

    /**
     * Get the latest manual trip count for this driver.
     */
    public function latestManualTripCount()
    {
        return $this->manualTripCounts()->latest()->first();
    }

    /**
     * Get the total number of trips (retase) for this driver including manual counts.
     */
    public function totalTripCount()
    {
        $manualCount = $this->latestManualTripCount();
        $systemCount = $this->tripCount();
        
        return $systemCount + ($manualCount ? ($manualCount->completed_trips + $manualCount->ongoing_trips) : 0);
    }

    /**
     * Get the total number of completed trips for this driver including manual counts.
     */
    public function totalCompletedTripCount()
    {
        $manualCount = $this->latestManualTripCount();
        $systemCount = $this->completedTripCount();
        
        return $systemCount + ($manualCount ? $manualCount->completed_trips : 0);
    }

    /**
     * Get the total number of ongoing trips for this driver including manual counts.
     */
    public function totalOngoingTripCount()
    {
        $manualCount = $this->latestManualTripCount();
        $systemCount = $this->ongoingTripCount();
        
        return $systemCount + ($manualCount ? $manualCount->ongoing_trips : 0);
    }

    /**
     * Get the user associated with the driver.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'id');
    }
}