<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'group',
        'type',
        'description',
        'help_text'
    ];

    // Perbarui metode set
    public static function set(string $key, $value, string $group = 'general', string $type = 'text', ?string $description = null, ?string $help_text = null)
    {
        $setting = self::where('key', $key)->first();
        
        if ($setting) {
            return $setting->update([
                'value' => $value,
                'group' => $group,
                'type' => $type,
                'description' => $description ?? $setting->description,
                'help_text' => $help_text ?? $setting->help_text
            ]);
        }
        
        return self::create([
            'key' => $key,
            'value' => $value,
            'group' => $group,
            'type' => $type,
            'description' => $description,
            'help_text' => $help_text
        ]) ? true : false;
    }

    /**
     * Get setting value by key
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function get(string $key, $default = null)
    {
        $setting = self::where('key', $key)->first();
        
        if (!$setting) {
            return $default;
        }
        
        return self::formatValue($setting->value, $setting->type);
    }

    /**
     * Format value based on type
     *
     * @param mixed $value
     * @param string $type
     * @return mixed
     */
    protected static function formatValue($value, string $type)
    {
        switch ($type) {
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            case 'number':
                return is_numeric($value) ? (strpos($value, '.') !== false ? (float) $value : (int) $value) : 0;
            case 'array':
            case 'json':
                return json_decode($value, true) ?? [];
            default:
                return $value;
        }
    }
}