<?php

namespace App\Services;

use App\Models\TruckMaintenance;
use App\Models\Driver;
use App\Models\User;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Google\Client as GoogleClient;

class NotificationService
{
    private $projectId = 'astro-studio-2b611'; // Dari google-services.json
    private $credentialsPath;
    
    public function __construct()
    {
        // Path ke file service account JSON
        $this->credentialsPath = storage_path('app/astro-studio-2b611-firebase-adminsdk-fbsvc-f29a47c0ad.json');
    }
    
    /**
     * Mendapatkan OAuth 2.0 access token untuk FCM v1 API
     */
    private function getAccessToken()
    {
        try {
            $client = new GoogleClient();
            $client->setAuthConfig($this->credentialsPath);
            $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
            
            // Nonaktifkan verifikasi SSL
            $httpClient = new \GuzzleHttp\Client([
                'verify' => false
            ]);
            $client->setHttpClient($httpClient);
            
            $client->refreshTokenWithAssertion();
            $token = $client->getAccessToken();
            
            return $token['access_token'];
        } catch (\Exception $e) {
            Log::error('FCM Access Token Error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Mendapatkan HTTP client dengan header otorisasi yang benar
     */
    private function getHttpClient()
    {
        $accessToken = $this->getAccessToken();
        
        if (!$accessToken) {
            Log::error('Failed to get FCM access token');
            return null;
        }
        
        return Http::withOptions([
            'verify' => false, // Disable SSL verification untuk development
        ])->withHeaders([
            'Authorization' => 'Bearer ' . $accessToken,
            'Content-Type' => 'application/json'
        ]);
    }

    public function sendManualPushNotification(TruckMaintenance $maintenance, $message = null)
    {
        try {
            $defaultMessage = "Pengingat maintenance {$maintenance->maintenance_type} untuk {$maintenance->truck->license_plate}";
            $notificationMessage = $message ?? $defaultMessage;
            
            $response = $this->sendFCMNotification($maintenance, $notificationMessage);
            
            return [
                'success' => $response && $response->successful(),
                'type' => 'push_notification',
                'response' => $response ? $response->json() : null
            ];
        } catch (\Exception $e) {
            Log::error('FCM Send Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function sendManualInAppMessage(TruckMaintenance $maintenance, $message = null)
    {
        // FCM tidak memiliki in-app message terpisah seperti OneSignal
        // Gunakan data payload untuk trigger in-app behavior
        return $this->sendManualPushNotification($maintenance, $message);
    }
    
    private function sendFCMNotification(TruckMaintenance $maintenance, $message)
    {
        $client = $this->getHttpClient();
        
        if (!$client) {
            return null;
        }
        
        // Format payload untuk FCM v1 API
        $payload = [
            'message' => [
                'topic' => 'drivers', // Untuk mengirim ke topic
                'notification' => [
                    'title' => 'Pengingat Maintenance',
                    'body' => $message
                ],
                'data' => [
                    'maintenance_id' => (string)$maintenance->id,
                    'truck_id' => (string)$maintenance->truck_id,
                    'type' => 'maintenance_reminder_manual'
                ],
                'android' => [
                    'priority' => 'high'
                ]
            ]
        ];
        
        Log::info('FCM v1 Request Payload:', $payload);
        
        // URL endpoint untuk FCM v1 API
        $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";
        $response = $client->post($url, $payload);
        
        Log::info('FCM v1 Response Status: ' . $response->status());
        Log::info('FCM v1 Response Body: ' . $response->body());
        
        return $response;
    }

    /**
     * Send FCM notification to a specific device using v1 API
     *
     * @param string $token FCM token of the device
     * @param array $data Data to send
     * @return bool
     */
    public function sendFCMNotificationToDevice($token, $data)
    {
        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                return false;
            }
            
            $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";
            
            // Format payload untuk FCM v1 API
            $payload = [
                'message' => [
                    'token' => $token,
                    'data' => array_map('strval', $data) // Pastikan semua nilai adalah string
                ]
            ];
            
            $headers = [
                'Authorization: Bearer ' . $accessToken,
                'Content-Type: application/json'
            ];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
            
            $result = curl_exec($ch);
            if ($result === FALSE) {
                Log::error('FCM v1 Send Error: ' . curl_error($ch));
                curl_close($ch);
                return false;
            }
            
            curl_close($ch);
            Log::info('FCM v1 Response: ' . $result);
            return true;
        } catch (\Exception $e) {
            Log::error('FCM v1 Exception: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Send maintenance notification to a specific driver
     *
     * @param int $driverId
     * @param int $maintenanceId
     * @param int $truckId
     * @return bool
     */
    public function sendMaintenanceNotificationToDriver($driverId, $maintenanceId, $truckId)
    {
        // Get the user associated with the driver
        $driver = Driver::find($driverId);
        if (!$driver || !$driver->user_id) {
            Log::error("Driver not found or not associated with a user: {$driverId}");
            return false;
        }
        
        $user = User::find($driver->user_id);
        if (!$user || !$user->fcm_token) {
            Log::error("User not found or FCM token not available for driver: {$driverId}");
            return false;
        }
        
        $data = [
            'type' => 'maintenance_reminder_manual',
            'maintenance_id' => $maintenanceId,
            'truck_id' => $truckId,
        ];
        
        return $this->sendFCMNotificationToDevice($user->fcm_token, $data);
    }
}