<?php

namespace App\Http\Controllers;

use App\Models\FuelPurchase;
use App\Models\Expense;
use App\Models\Truck;
use App\Models\Driver;
use App\Models\FuelStock;
use App\Models\FuelStockLog; // Tambahkan import ini
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class FuelPurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Filter berdasarkan pencarian dan tanggal
        $query = FuelPurchase::with('truck');
        
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->whereHas('truck', function($truck) use ($search) {
                    $truck->where('plate_number', 'like', "%{$search}%");
                })->orWhere('fuel_type', 'like', "%{$search}%");
            });
        }
        
        if ($request->filled('start_date')) {
            $query->where('purchase_date', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->where('purchase_date', '<=', $request->end_date);
        }
        
        $fuelPurchases = $query->latest()->paginate(10);
        
        // Ambil data stok BBM
        $fuelStocks = FuelStock::all();
        
        // Ambil log operasi manual stok BBM
        $fuelStockLogs = FuelStockLog::with('user')
            ->latest('operation_date')
            ->limit(10)
            ->get();
        
        // Hitung total pembelian BBM per jenis
        $fuelPurchasesByType = FuelPurchase::select('fuel_type', DB::raw('SUM(quantity) as total_quantity'))
            ->groupBy('fuel_type')
            ->get()
            ->keyBy('fuel_type');
        
        // Hitung total pengeluaran Solar bulan berjalan dalam liter
        $solarUsageCurrentMonth = FuelPurchase::where('fuel_type', 'Solar')
            ->whereYear('purchase_date', now()->year)
            ->whereMonth('purchase_date', now()->month)
            ->sum('quantity');
        
        // Tambahkan operasi manual pengurangan stok Solar bulan berjalan
        $solarManualReduction = FuelStockLog::where('fuel_type', 'Solar')
            ->where('operation_type', 'reduce')
            ->whereYear('operation_date', now()->year)
            ->whereMonth('operation_date', now()->month)
            ->sum('quantity');
        
        $solarUsageCurrentMonth += $solarManualReduction;
        
        return view('fuel-purchases.index', compact(
            'fuelPurchases', 
            'fuelStocks', 
            'fuelPurchasesByType', 
            'solarUsageCurrentMonth',
            'fuelStockLogs'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $trucks = Truck::where('status', 'active')->get();
        $drivers = Driver::orderBy('name')->get();
        return view('fuel-purchases.create', compact('trucks', 'drivers'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'truck_id' => 'required|exists:trucks,id',
            'purchase_date' => 'required|date',
            'quantity' => 'required|numeric|min:0',
            'price_per_liter' => 'required|numeric|min:0',
            'total_cost' => 'required|numeric|min:0',
            'fuel_type' => 'required|string|max:255',
            'station_name' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'driver_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'receipt_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
        
        $data = $request->all();
        
        // Handle receipt image upload
        if ($request->hasFile('receipt_image')) {
            $path = $request->file('receipt_image')->store('receipts', 'public');
            $data['receipt_image'] = $path;
        }
        
        DB::beginTransaction();
        
        try {
            // Buat record pembelian BBM
            $fuelPurchase = FuelPurchase::create($data);
            
            // Update stok BBM
            FuelStock::addStock($data['fuel_type'], $data['quantity']);
            
            DB::commit();
            
            return redirect()->route('fuel-purchases.index')
                ->with('success', 'Pengisian bahan bakar berhasil ditambahkan dan stok diperbarui.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(FuelPurchase $fuelPurchase)
    {
        return view('fuel-purchases.show', compact('fuelPurchase'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FuelPurchase $fuelPurchase)
    {
        $validated = $request->validate([
            'truck_id' => 'required|exists:trucks,id',
            'purchase_date' => 'required|date',
            'quantity' => 'required|numeric|min:0',
            'price_per_liter' => 'required|numeric|min:0',
            'total_cost' => 'required|numeric|min:0',
            'fuel_type' => 'required|string|max:255',
            'station_name' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'driver_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'receipt_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
        
        $data = $request->all();
        
        // Handle receipt image upload
        if ($request->hasFile('receipt_image')) {
            // Delete old image if exists
            if ($fuelPurchase->receipt_image) {
                Storage::disk('public')->delete($fuelPurchase->receipt_image);
            }
            
            $path = $request->file('receipt_image')->store('receipts', 'public');
            $data['receipt_image'] = $path;
        }
        
        DB::beginTransaction();
        
        try {
            // Kurangi stok lama
            FuelStock::reduceStock($fuelPurchase->fuel_type, $fuelPurchase->quantity);
            
            // Update record pembelian BBM
            $fuelPurchase->update($data);
            
            // Tambahkan stok baru
            FuelStock::addStock($data['fuel_type'], $data['quantity']);
            
            DB::commit();
            
            return redirect()->route('fuel-purchases.index')
                ->with('success', 'Pengisian bahan bakar berhasil diperbarui dan stok diperbarui.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FuelPurchase $fuelPurchase)
    {
        DB::beginTransaction();
        
        try {
            // Kurangi stok
            FuelStock::reduceStock($fuelPurchase->fuel_type, $fuelPurchase->quantity);
            
            // Hapus record
            $fuelPurchase->delete();
            
            DB::commit();
            
            return redirect()->route('fuel-purchases.index')
                ->with('success', 'Data pembelian bahan bakar berhasil dihapus dan stok diperbarui.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }
    
    // Sisanya tetap sama seperti sebelumnya...

    /**
     * Handle stock operations (add/reduce)
     */
    public function handleStockOperation(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:add_stock,reduce_stock',
            'fuel_type' => 'required|string',
            'quantity' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);
        
        DB::beginTransaction();
        
        try {
            if ($request->action === 'add_stock') {
                // Tambah stok dengan logging
                FuelStock::addStock($request->fuel_type, $request->quantity, $request->notes, true);
                $message = 'Stok BBM berhasil ditambahkan.';
            } else {
                // Kurangi stok dengan logging
                FuelStock::reduceStock($request->fuel_type, $request->quantity, $request->notes, true);
                $message = 'Stok BBM berhasil dikurangi.';
            }
            
            DB::commit();
            
            return redirect()->route('fuel-purchases.index')
                ->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }
}
