<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Trip;
use App\Models\Truck;
use App\Models\Driver;
use App\Models\Project;
use App\Models\Expense;
use App\Models\Income;
use App\Models\FuelPurchase;
use App\Models\Client;
use Barryvdh\DomPDF\Facade\Pdf;
use Rap2hpoutre\FastExcel\FastExcel;
use Carbon\Carbon;

class ReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:view-reports');
    }

    // =================== TRIPS ===================
    public function trips(Request $request)
    {
        $data = $this->tripsData($request);
        return view('reports.trips', $data);
    }
    public function exportTripsPDF(Request $request)
    {
        $data = $this->tripsData($request);
        return Pdf::loadView('reports.trips_pdf', $data)->download('laporan_perjalanan.pdf');
    }
    public function exportTripsExcel(Request $request)
    {
        $data = $this->tripsData($request);
        $trips = $data['trips'];
        $export = $trips->map(function($trip) {
            return [
                'Tanggal' => $trip->start_date ? $trip->start_date->format('d/m/Y') : '',
                'Plat Nomor' => $this->getPlatNomor($trip),
                'Driver' => optional($trip->driver)->name ?? '-',
                'Username Driver' => optional($trip->driver)->username ?? '-',
                'Dari' => $trip->origin,
                'Tujuan' => $trip->destination,
                'Jarak (km)' => $trip->distance,
                'Retase' => $trip->driverActivities->sum('retase'),
                'Material' => optional($trip->materialNp)->name ?? '-',
                'Proyek' => optional($trip->project)->name ?? '-',
            ];
        });
        return (new FastExcel($export))->download('laporan_perjalanan.xlsx');
    }
    private function getPlatNomor($trip)
    {
        if (isset($trip->truck) && $trip->truck) {
            return $trip->truck->plate_number;
        } elseif (isset($trip->driver) && isset($trip->driver->truck)) {
            return $trip->driver->truck->plate_number ?? '-';
        } elseif (isset($trip->driver) && isset($trip->driver->trucks) && $trip->driver->trucks->count()) {
            return $trip->driver->trucks->first()->plate_number ?? '-';
        } else {
            return '-';
        }
    }

    private function tripsData(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $license = $request->input('license_plate');
        $driver = $request->input('driver_id');
        $project = $request->input('project_id');

        $trips = Trip::with(['truck', 'driver', 'project'])
            ->whereBetween('start_date', [$start, $end])
            ->when($license, fn($q) => $q->whereHas('truck', fn($q2) => $q2->where('plate_number', 'like', "%$license%")))
            ->when($driver, fn($q) => $q->where('driver_id', $driver))
            ->when($project, fn($q) => $q->where('project_id', $project))
            ->get();

        return [
            'trips' => $trips,
            'totalDistance' => $trips->sum('distance'),
            'totalLoad' => $trips->sum('load_weight'),
            'startDate' => $start,
            'endDate' => $end,
            'licensePlate' => $license,
            'driverId' => $driver,
            'projectId' => $project,
            'drivers' => Driver::all(),
            'projects' => Project::all(),
            'tripsByTruck' => $trips->groupBy('truck_id')->map->count(),
            'tripsByProject' => $trips->groupBy('project_id')->map->count(),
        ];
    }

    // =================== FUEL ===================
    public function fuel(Request $request)
    {
        $data = $this->fuelData($request);
        return view('reports.fuel', $data);
    }
    public function exportFuelPDF(Request $request)
    {
        $data = $this->fuelData($request);
        return Pdf::loadView('reports.fuel_pdf', $data)->download('laporan_bahan_bakar.pdf');
    }
    public function exportFuelExcel(Request $request)
    {
        $data = $this->fuelData($request);
        return (new FastExcel($data['fuelPurchases']))->download('laporan_bahan_bakar.xlsx');
    }
    private function fuelData(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truck = $request->input('truck_id');
        $fuelType = $request->input('fuel_type');
        // Hapus filter location karena tidak ada di tabel
        // $location = $request->input('location');
    
        $fuels = FuelPurchase::with('truck')
            ->whereBetween('purchase_date', [$start, $end])
            ->when($truck, fn($q) => $q->where('truck_id', $truck))
            ->when($fuelType, fn($q) => $q->where('fuel_type', $fuelType))
            // ->when($location, fn($q) => $q->where('location', 'like', "%$location%"))
            ->get();
    
        // Process chart data with numeric validation
        $fuelByTruck = $fuels->groupBy('truck_id')->map->sum('quantity');
        $fuelByMonth = $fuels->groupBy(function($item) {
            return \Carbon\Carbon::parse($item->purchase_date)->format('Y-m');
        })->map->sum('total_cost');
    
        // Pastikan data selalu numerik
        $fuelByTruck = $fuelByTruck->map(function($volume) {
            return (float) $volume;
        });
    
        $fuelByMonth = $fuelByMonth->map(function($cost) {
            return (float) $cost;
        });
    
        return [
            'fuelPurchases' => $fuels,
            'totalVolume' => $fuels->sum('quantity'), // Ganti volume ke quantity
            'totalCost' => $fuels->sum('total_cost'),
            'startDate' => $start,
            'endDate' => $end,
            'truckId' => $truck,
            'fuelType' => $fuelType,
            // 'location' => $location,
            'trucks' => Truck::all(),
            'fuelTypes' => FuelPurchase::distinct()->pluck('fuel_type'),
            'fuelByTruck' => $fuelByTruck,
            'fuelByMonth' => $fuelByMonth,
        ];
    }

    // =================== EXPENSES ===================
    public function expenses(Request $request)
    {
        $data = $this->expensesData($request);
        return view('reports.expenses', $data);
    }
    public function exportExpensesPDF(Request $request)
    {
        $data = $this->expensesData($request);
        return Pdf::loadView('reports.expenses_pdf', $data)->download('laporan_pengeluaran.pdf');
    }
    public function exportExpensesExcel(Request $request)
    {
        $data = $this->expensesData($request);
        return (new FastExcel($data['expenses']))->download('laporan_pengeluaran.xlsx');
    }
    private function expensesData(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $project = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');

        $expenses = Expense::with(['project', 'addedBy'])
            ->whereBetween('expense_date', [$start, $end])
            ->when($project, fn($q) => $q->where('project_id', $project))
            ->when($category, fn($q) => $q->where('category', $category))
            ->when($status, fn($q) => $q->where('status', $status))
            ->get();

        return [
            'expenses' => $expenses,
            'totalExpenses' => $expenses->sum('amount'),
            'startDate' => $start,
            'endDate' => $end,
            'projectId' => $project,
            'category' => $category,
            'status' => $status,
            'projects' => Project::all(),
            'categories' => Expense::distinct()->pluck('category'),
            'statuses' => ['draft', 'approved', 'rejected'],
            'expensesByCategory' => $expenses->groupBy('category')->map->sum('amount'),
        ];
    }

    // =================== PROJECTS ===================
    public function projects(Request $request)
    {
        $data = $this->projectsData($request);
        return view('reports.projects', $data);
    }
    public function exportProjectsPDF(Request $request)
    {
        $data = $this->projectsData($request);
        return Pdf::loadView('reports.projects_pdf', $data)->download('laporan_proyek.pdf');
    }
    public function exportProjectsExcel(Request $request)
    {
        $data = $this->projectsData($request);
        return (new FastExcel($data['projectStats']))->download('laporan_proyek.xlsx');
    }
    private function projectsData(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $client = $request->input('client_id');

        $projects = Project::with('client')
            ->when($client, fn($q) => $q->where('client_id', $client))
            ->get();

        $projectStats = $projects->map(function($project) use ($start, $end) {
            $trips = Trip::where('project_id', $project->id)->whereBetween('start_date', [$start, $end])->get();
            $expenses = Expense::where('project_id', $project->id)->whereBetween('expense_date', [$start, $end])->get();
            $incomes = Income::where('project_id', $project->id)->whereBetween('income_date', [$start, $end])->get();
            return [
                'project' => $project,
                'trips_count' => $trips->count(),
                'total_distance' => $trips->sum('distance'),
                'total_load' => $trips->sum('load_weight'),
                'expenses_total' => $expenses->sum('amount'),
                'incomes_total' => $incomes->sum('amount'),
                'profit' => $incomes->sum('amount') - $expenses->sum('amount')
            ];
        });

        return [
            'projectStats' => $projectStats,
            'startDate' => $start,
            'endDate' => $end,
            'clientId' => $client,
            'clients' => Client::all(),
        ];
    }

    // =================== FINANCIAL ===================
    public function financial(Request $request)
    {
        $data = $this->financialData($request);
        return view('reports.financial', $data);
    }
    public function exportFinancialPDF(Request $request)
    {
        $data = $this->financialData($request);
        return Pdf::loadView('reports.financial_pdf', $data)->download('laporan_keuangan.pdf');
    }
    public function exportFinancialExcel(Request $request)
    {
        $data = $this->financialData($request);
        return (new FastExcel($data['all']))->download('laporan_keuangan.xlsx');
    }
    private function financialData(Request $request)
    {
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $project = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');

        $expenses = Expense::with(['project', 'addedBy'])
            ->whereBetween('expense_date', [$start, $end])
            ->when($project, fn($q) => $q->where('project_id', $project))
            ->when($category, fn($q) => $q->where('category', $category))
            ->when($status, fn($q) => $q->where('status', $status))
            ->get();

        $incomes = Income::with(['project', 'addedBy'])
            ->whereBetween('income_date', [$start, $end])
            ->when($project, fn($q) => $q->where('project_id', $project))
            ->when($category, fn($q) => $q->where('category', $category))
            ->get();

        $totalExpenses = $expenses->sum('amount');
        $totalIncomes = $incomes->sum('amount');
        $balance = $totalIncomes - $totalExpenses;

        $all = collect();
        foreach ($incomes as $income) {
            $all->push([
                'Tipe' => 'Pendapatan',
                'Judul' => $income->title,
                'Jumlah (Rp)' => number_format($income->amount, 2, ',', '.'),
                'Tanggal' => $income->income_date->format('d/m/Y'),
                'Kategori' => $income->category,
                'Proyek' => $income->project ? $income->project->name : 'Tidak Ada',
                'Status' => '-',
                'Ditambahkan Oleh' => $income->addedBy->name ?? '-',
            ]);
        }
        foreach ($expenses as $expense) {
            $all->push([
                'Tipe' => 'Pengeluaran',
                'Judul' => $expense->title,
                'Jumlah (Rp)' => number_format($expense->amount, 2, ',', '.'),
                'Tanggal' => $expense->expense_date->format('d/m/Y'),
                'Kategori' => $expense->category,
                'Proyek' => $expense->project ? $expense->project->name : 'Tidak Ada',
                'Status' => ucfirst($expense->status),
                'Ditambahkan Oleh' => $expense->addedBy->name ?? '-',
            ]);
        }

        return [
            'expenses' => $expenses,
            'incomes' => $incomes,
            'totalExpenses' => $totalExpenses,
            'totalIncomes' => $totalIncomes,
            'balance' => $balance,
            'startDate' => $start,
            'endDate' => $end,
            'projectId' => $project,
            'category' => $category,
            'status' => $status,
            'projects' => Project::all(),
            'categories' => Expense::distinct()->pluck('category')->merge(Income::distinct()->pluck('category'))->unique(),
            'statuses' => ['draft', 'approved', 'rejected'],
            'all' => $all,
        ];
    }
}