<?php

namespace App\Http\Controllers;

use App\Models\Truck;
use App\Models\TruckMaintenance;
use Illuminate\Http\Request;

use App\Services\NotificationService;

class TruckMaintenanceController extends Controller
{
    public function index(Truck $truck)
    {
        $maintenances = $truck->maintenances()
            ->when(request('search'), function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('maintenance_type', 'like', "%{$search}%")
                      ->orWhere('location', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->when(request('status'), function ($query, $status) {
                $query->where('status', $status);
            })
            ->latest()
            ->paginate(10);

        return view('truck-maintenances.index', compact('truck', 'maintenances'));
    }

    public function create(Truck $truck)
    {
        return view('truck-maintenances.create', compact('truck'));
    }

    public function store(Request $request, Truck $truck)
    {
        $validated = $request->validate([
            'maintenance_date' => 'required|date',
            'maintenance_type' => 'required|string',
            'description' => 'required|string',
            'cost' => 'required|numeric',
            'location' => 'required|string',
            'performed_by' => 'required|string',
            'status' => 'required|in:' . TruckMaintenance::getValidStatusesString(),
            'odometer_reading' => 'nullable|numeric',
            'next_maintenance_date' => 'nullable|date',
            'notes' => 'nullable|string'
        ]);

        $truck->maintenances()->create($validated);

        return redirect()
            ->route('truck-maintenances.index', $truck)
            ->with('success', 'Data maintenance berhasil ditambahkan.');
    }

    public function edit(Truck $truck, TruckMaintenance $maintenance)
    {
        return view('truck-maintenances.edit', compact('truck', 'maintenance'));
    }

    public function update(Request $request, Truck $truck, TruckMaintenance $maintenance)
    {
        $validated = $request->validate([
            'maintenance_date' => 'required|date',
            'maintenance_type' => 'required|string',
            'description' => 'required|string',
            'cost' => 'required|numeric',
            'location' => 'required|string',
            'performed_by' => 'required|string',
            'status' => 'required|in:pending,in_progress,completed,scheduled,draft',
            'odometer_reading' => 'nullable|numeric',
            'next_maintenance_date' => 'nullable|date',
            'notes' => 'nullable|string'
        ]);

        $maintenance->update($validated);

        return redirect()
            ->route('truck-maintenances.index', $truck)
            ->with('success', 'Data maintenance berhasil diperbarui.');
    }

    public function destroy(Truck $truck, TruckMaintenance $maintenance)
    {
        $maintenance->delete();

        return redirect()
            ->route('truck-maintenances.index', $truck)
            ->with('success', 'Data maintenance berhasil dihapus.');
    }

    public function allMaintenances()
    {
        $maintenances = TruckMaintenance::with('truck')
            ->when(request('search'), function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('maintenance_type', 'like', "%{$search}%")
                      ->orWhere('location', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->when(request('status'), function ($query, $status) {
                $query->where('status', $status);
            })
            ->latest()
            ->paginate(10);

        return view('truck-maintenances.all', compact('maintenances'));
    }

    public function show(Truck $truck, TruckMaintenance $maintenance)
    {
        $maintenance->load('requestedBy');
        return view('truck-maintenances.show', compact('truck', 'maintenance'));
    }

    public function sendManualPushNotification(Request $request, Truck $truck, TruckMaintenance $maintenance)
    {
        $request->validate([
            'message' => 'nullable|string|max:255'
        ]);
        
        $notificationService = new NotificationService();
        $result = $notificationService->sendManualPushNotification(
            $maintenance, 
            $request->input('message')
        );
        
        if ($result['success']) {
            // Log manual push notification
            $maintenance->update([
                'last_manual_push_sent_at' => now(),
                'manual_push_count' => $maintenance->manual_push_count + 1
            ]);
            
            return redirect()->back()->with('success', 'Push notification berhasil dikirim ke aplikasi mobile!');
        } else {
            return redirect()->back()->with('error', 'Gagal mengirim push notification: ' . ($result['error'] ?? 'Unknown error'));
        }
    }
    
    public function sendManualInAppMessage(Request $request, Truck $truck, TruckMaintenance $maintenance)
    {
        $request->validate([
            'message' => 'nullable|string|max:255'
        ]);
        
        $notificationService = new NotificationService();
        $result = $notificationService->sendManualInAppMessage(
            $maintenance, 
            $request->input('message')
        );
        
        if ($result['success']) {
            // Log manual in-app message
            $maintenance->update([
                'last_manual_inapp_sent_at' => now(),
                'manual_inapp_count' => $maintenance->manual_inapp_count + 1
            ]);
            
            return redirect()->back()->with('success', 'In-app message berhasil dikirim ke aplikasi mobile!');
        } else {
            return redirect()->back()->with('error', 'Gagal mengirim in-app message: ' . ($result['error'] ?? 'Unknown error'));
        }
    }
    
    public function sendBulkNotification(Request $request)
    {
        $request->validate([
            'maintenance_ids' => 'required|array',
            'maintenance_ids.*' => 'exists:truck_maintenances,id',
            'message' => 'nullable|string|max:255'
        ]);
        
        $notificationService = new NotificationService();
        $successCount = 0;
        $failCount = 0;
        
        foreach ($request->maintenance_ids as $maintenanceId) {
            $maintenance = TruckMaintenance::find($maintenanceId);
            $result = $notificationService->sendManualMaintenanceNotification(
                $maintenance, 
                $request->input('message')
            );
            
            if ($result['success']) {
                $successCount++;
                $maintenance->update([
                    'last_manual_notification_sent_at' => now(),
                    'manual_notification_count' => $maintenance->manual_notification_count + 1
                ]);
            } else {
                $failCount++;
            }
        }
        
        return redirect()->back()->with('success', "Berhasil mengirim {$successCount} notifikasi, gagal {$failCount} notifikasi.");
    }

    /**
     * Submit maintenance request from mobile app
     */
    public function submitMaintenanceRequestApi(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
            ]);
            
            $user = auth()->user();
            
            // Jika user adalah driver, ambil truck_id dari driver
            $truckId = null;
            if ($user->driver && $user->driver->truck_id) {
                $truckId = $user->driver->truck_id;
            }
            
            $imagePath = null;
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('maintenance-requests', 'public');
            }
            
            $maintenance = TruckMaintenance::create([
                'truck_id' => $truckId,
                'maintenance_type' => $validated['title'],
                'description' => $validated['description'],
                'status' => 'draft', // Status draft untuk review
                'maintenance_date' => now(),
                'cost' => 0,
                'location' => 'Request dari Driver',
                'performed_by' => $user->name ?? 'Driver',
                'notes' => 'Request maintenance dari aplikasi mobile',
                'image_path' => $imagePath,
                'requested_by' => $user->id,
                'requested_at' => now()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Request maintenance berhasil dikirim',
                'data' => $maintenance
            ], 201);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengirim request maintenance',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get maintenance schedule for mobile API
     */
    public function getMaintenanceScheduleApi()
    {
        // Logging header Authorization untuk debug
        \Log::info('Sanctum raw header', [
            'auth_header' => request()->header('Authorization')
        ]);

        try {
            $user = auth()->user();
            $driver = $user ? $user->driver : null;
            $truckId = $driver ? $driver->truck_id : null;

            \Log::info('API getMaintenanceScheduleApi', [
                'user_id' => $user ? $user->id : null,
                'driver_id' => $driver ? $driver->id : null,
                'truck_id' => $truckId,
            ]);

            if (!$user) {
                \Log::warning('User tidak ditemukan saat akses API maintenance');
                return response()->json([
                    'error' => 'User tidak ditemukan atau tidak login',
                ], 401);
            }

            if (!$driver) {
                \Log::warning('User tidak punya relasi driver', [
                    'user_id' => $user->id
                ]);
                return response()->json([
                    'error' => 'User ini bukan driver atau belum terhubung ke data driver',
                ], 404);
            }

            if (!$truckId) {
                \Log::warning('Driver tidak punya truck', [
                    'driver_id' => $driver->id
                ]);
                return response()->json([
                    'error' => 'Driver belum memiliki truck',
                ], 404);
            }

            $maintenances = TruckMaintenance::with('truck')
                ->where('truck_id', $truckId)
                ->where('status', '!=', 'draft')
                ->orderBy('next_maintenance_date', 'desc')  // Ubah dari 'asc' ke 'desc' untuk tanggal terbaru di atas
                ->get();

            \Log::info('Maintenance count', [
                'truck_id' => $truckId,
                'maintenance_count' => $maintenances->count()
            ]);

            $formattedMaintenances = $maintenances->map(function ($maintenance) {
                return [
                    'id' => $maintenance->id ?? 0,
                    'truckId' => $maintenance->truck_id ?? 0,
                    'licensePlate' => (
                        !isset($maintenance->truck) || !$maintenance->truck
                            ? 'Truck tidak ditemukan'
                            : (empty($maintenance->truck->plate_number)  // Ubah dari license_plate ke plate_number
                                ? 'Belum diisi'
                                : $maintenance->truck->plate_number)     // Ubah dari license_plate ke plate_number
                    ),
                    'maintenanceType' => $maintenance->maintenance_type ?? '-',
                    'nextMaintenanceDate' => $maintenance->next_maintenance_date ? $maintenance->next_maintenance_date->format('Y-m-d') : (($maintenance->maintenance_date) ? $maintenance->maintenance_date->format('Y-m-d') : '-'),
                    'description' => $maintenance->description ?? '-',
                    'status' => $maintenance->status ?? '-'
                ];
            });

            return response()->json($formattedMaintenances);
        } catch (\Exception $e) {
            \Log::error('API getMaintenanceScheduleApi error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'error' => 'Gagal mengambil data maintenance',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send manual notification to driver
     */
    public function sendManualNotification(TruckMaintenance $maintenance)
    {
        // Validate that the maintenance has a truck
        if (!$maintenance->truck_id) {
            return redirect()->back()->with('error', 'Maintenance tidak memiliki truck yang terkait.');
        }
        
        $truck = Truck::find($maintenance->truck_id);
        if (!$truck || !$truck->driver_id) {
            return redirect()->back()->with('error', 'Truck tidak ditemukan atau tidak memiliki driver yang terkait.');
        }
        
        // Send notification
        $notificationService = new NotificationService();
        $result = $notificationService->sendMaintenanceNotificationToDriver(
            $truck->driver_id,
            $maintenance->id,
            $truck->id
        );
        
        if ($result) {
            // Update notification sent status
            $maintenance->manual_notification_sent = true;
            $maintenance->manual_notification_sent_at = now();
            $maintenance->save();
            
            return redirect()->back()->with('success', 'Notifikasi berhasil dikirim ke driver.');
        }
        
        return redirect()->back()->with('error', 'Gagal mengirim notifikasi ke driver.');
    }
}
