<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\FuelStockLog;

class FuelStock extends Model
{
    protected $fillable = [
        'fuel_type',
        'current_stock',
        'initial_stock',
        'last_updated',
        'notes'
    ];

    protected $casts = [
        'current_stock' => 'decimal:2',
        'initial_stock' => 'decimal:2',
        'last_updated' => 'date',
    ];

    /**
     * Mendapatkan stok berdasarkan jenis BBM
     */
    public static function getStockByType($fuelType)
    {
        return self::where('fuel_type', $fuelType)->first();
    }

    /**
     * Menambahkan stok BBM dengan logging
     */
    public static function addStock($fuelType, $quantity, $notes = null, $isManual = false)
    {
        $stock = self::getStockByType($fuelType);
        
        if (!$stock) {
            $stock = self::create([
                'fuel_type' => $fuelType,
                'current_stock' => $quantity,
                'initial_stock' => $quantity,
                'last_updated' => now(),
            ]);
        } else {
            $stock->current_stock += $quantity;
            $stock->last_updated = now();
            $stock->save();
        }
        
        // Log operasi manual
        if ($isManual) {
            FuelStockLog::create([
                'fuel_type' => $fuelType,
                'operation_type' => 'add',
                'quantity' => $quantity,
                'notes' => $notes,
                'created_by' => auth()->id(),
                'operation_date' => now(),
            ]);
        }
        
        return $stock;
    }

    /**
     * Mengurangi stok BBM dengan logging
     */
    public static function reduceStock($fuelType, $quantity, $notes = null, $isManual = false)
    {
        $stock = self::getStockByType($fuelType);
        
        if (!$stock) {
            $stock = self::create([
                'fuel_type' => $fuelType,
                'current_stock' => -$quantity,
                'initial_stock' => 0,
                'last_updated' => now(),
            ]);
        } else {
            $stock->current_stock -= $quantity;
            $stock->last_updated = now();
            $stock->save();
        }
        
        // Log operasi manual
        if ($isManual) {
            FuelStockLog::create([
                'fuel_type' => $fuelType,
                'operation_type' => 'reduce',
                'quantity' => $quantity,
                'notes' => $notes,
                'created_by' => auth()->id(),
                'operation_date' => now(),
            ]);
        }
        
        return $stock;
    }
}
