<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Truck extends Model
{
    use HasFactory;
    
    protected $fillable = [
        'plate_number', 'brand', 'model', 'year', 'capacity',
        'fuel_type', 'status', 'purchase_date', 'last_service_date',
        'next_service_date', 'stnk_expire_date', 'stnk_number',
        'kir_expire_date', 'kir_number', 'tax_expire_date', 'notes'
    ];

    protected $casts = [
        'purchase_date' => 'date',
        'last_service_date' => 'date',
        'next_service_date' => 'date',
        'stnk_expire_date' => 'date',
        'kir_expire_date' => 'date',
        'tax_expire_date' => 'date',
    ];

    /**
     * Get the trips associated with the truck.
     */
    public function trips(): HasMany
    {
        return $this->hasMany(Trip::class);
    }

    /**
     * Get the fuel purchases associated with the truck.
     */
    public function fuelPurchases(): HasMany
    {
        return $this->hasMany(FuelPurchase::class);
    }

    /**
     * Get the maintenances associated with the truck.
     */
    public function maintenances(): HasMany
    {
        return $this->hasMany(TruckMaintenance::class);
    }
    
    /**
     * Check if STNK is expired or about to expire
     */
    public function getStnkStatusAttribute()
    {
        if (!$this->stnk_expire_date) return null;
        
        $daysUntilExpire = now()->diffInDays($this->stnk_expire_date, false);
        
        if ($daysUntilExpire < 0) {
            return ['status' => 'overdue', 'days' => (int)abs($daysUntilExpire)]; // Dibulatkan menjadi integer
        } elseif ($daysUntilExpire <= 30) {
            return ['status' => 'warning', 'days' => (int)$daysUntilExpire]; // Dibulatkan menjadi integer
        } else {
            return ['status' => 'ok', 'days' => (int)$daysUntilExpire]; // Dibulatkan menjadi integer
        }
    }
    
    /**
     * Check if KIR is expired or about to expire
     */
    public function getKirStatusAttribute()
    {
        if (!$this->kir_expire_date) return null;
        
        $daysUntilExpire = now()->diffInDays($this->kir_expire_date, false);
        
        if ($daysUntilExpire < 0) {
            return ['status' => 'overdue', 'days' => (int)abs($daysUntilExpire)]; // Dibulatkan menjadi integer
        } elseif ($daysUntilExpire <= 30) {
            return ['status' => 'warning', 'days' => (int)$daysUntilExpire]; // Dibulatkan menjadi integer
        } else {
            return ['status' => 'ok', 'days' => (int)$daysUntilExpire]; // Dibulatkan menjadi integer
        }
    }
    
    /**
     * Check if Tax is expired or about to expire
     */
    public function getTaxStatusAttribute()
    {
        if (!$this->tax_expire_date) return null;
        
        $daysUntilExpire = now()->diffInDays($this->tax_expire_date, false);
        
        if ($daysUntilExpire < 0) {
            return ['status' => 'overdue', 'days' => (int)abs($daysUntilExpire)];
        } elseif ($daysUntilExpire <= 30) {
            return ['status' => 'warning', 'days' => (int)$daysUntilExpire];
        } else {
            return ['status' => 'ok', 'days' => (int)$daysUntilExpire];
        }
    }
}