<?php

namespace Database\Seeders;

use App\Models\Truck;
use App\Models\TruckMaintenance;
use Illuminate\Database\Seeder;
use Carbon\Carbon;

class TruckMaintenanceSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Dapatkan semua truk yang ada
        $trucks = Truck::all();
        
        // Jenis-jenis maintenance
        $maintenanceTypes = [
            'oil_change',
            'repair',
            'inspection',
            'tire_change',
            'filter_change',
            'brake_service',
            'electrical',
            'body_repair',
            'other'
        ];
        
        // Lokasi service
        $locations = [
            'Bengkel Resmi Hino',
            'Bengkel Resmi Mitsubishi',
            'Bengkel Resmi Isuzu',
            'Bengkel Mandiri Sejahtera',
            'Bengkel Jaya Abadi',
            'Bengkel Internal OBM',
            'Bengkel Mitra Sejati'
        ];
        
        // Teknisi/mekanik
        $performers = [
            'Tim Mekanik Internal',
            'Teknisi Resmi Dealer',
            'Budi Santoso',
            'Ahmad Wijaya',
            'Rudi Hermawan',
            'Joko Susilo',
            'Deni Pratama'
        ];
        
        // Status maintenance
        $statuses = ['scheduled', 'in_progress', 'completed'];
        
        // Untuk setiap truk, buat beberapa catatan maintenance
        foreach ($trucks as $truck) {
            // Buat 3-7 catatan maintenance untuk setiap truk
            $maintenanceCount = rand(3, 7);
            
            for ($i = 0; $i < $maintenanceCount; $i++) {
                // Tentukan tanggal maintenance (dalam 2 tahun terakhir)
                $maintenanceDate = Carbon::now()->subDays(rand(1, 730));
                
                // Pilih jenis maintenance secara acak
                $maintenanceType = $maintenanceTypes[array_rand($maintenanceTypes)];
                
                // Buat deskripsi berdasarkan jenis maintenance
                $description = $this->generateDescription($maintenanceType, $truck);
                
                // Tentukan biaya maintenance (antara 100rb - 10jt)
                $cost = rand(100000, 10000000);
                
                // Tentukan pembaca odometer (KM)
                $odometerReading = rand(10000, 100000);
                
                // Tentukan tanggal maintenance berikutnya (jika ada)
                $nextMaintenanceDate = null;
                if (in_array($maintenanceType, ['oil_change', 'inspection', 'filter_change'])) {
                    $nextMaintenanceDate = (clone $maintenanceDate)->addMonths(rand(3, 6));
                }
                
                // Tentukan status (lebih banyak completed)
                $status = rand(1, 10) <= 8 ? 'completed' : $statuses[array_rand($statuses)];
                
                // Buat catatan maintenance
                TruckMaintenance::create([
                    'truck_id' => $truck->id,
                    'maintenance_date' => $maintenanceDate,
                    'maintenance_type' => $maintenanceType,
                    'description' => $description,
                    'cost' => $cost,
                    'performed_by' => $performers[array_rand($performers)],
                    'location' => $locations[array_rand($locations)],
                    'odometer_reading' => $odometerReading,
                    'next_maintenance_date' => $nextMaintenanceDate,
                    'status' => $status,
                    'notes' => $this->generateNotes($status, $maintenanceType),
                ]);
            }
            
            // Tambahkan maintenance terjadwal untuk truk yang aktif
            if ($truck->status === 'active') {
                // Jadwalkan maintenance oli dalam 30 hari ke depan
                TruckMaintenance::create([
                    'truck_id' => $truck->id,
                    'maintenance_date' => Carbon::now()->addDays(rand(10, 30)),
                    'maintenance_type' => 'oil_change',
                    'description' => 'Penggantian oli dan filter rutin',
                    'cost' => rand(500000, 1500000),
                    'performed_by' => null,
                    'location' => $locations[array_rand($locations)],
                    'odometer_reading' => null,
                    'next_maintenance_date' => null,
                    'status' => 'scheduled',
                    'notes' => 'Jadwal maintenance rutin',
                ]);
            }
        }
    }
    
    /**
     * Generate deskripsi berdasarkan jenis maintenance
     */
    private function generateDescription($type, $truck)
    {
        $descriptions = [
            'oil_change' => [
                "Penggantian oli mesin dan filter oli untuk {$truck->brand} {$truck->model}",
                "Service rutin penggantian oli untuk truk {$truck->plate_number}",
                "Penggantian oli mesin, transmisi, dan filter"
            ],
            'repair' => [
                "Perbaikan sistem transmisi pada truk {$truck->plate_number}",
                "Perbaikan kebocoran sistem hidrolik",
                "Perbaikan sistem pengereman dan penggantian kampas rem",
                "Perbaikan sistem kemudi yang bermasalah"
            ],
            'inspection' => [
                "Inspeksi rutin 10.000 KM untuk truk {$truck->plate_number}",
                "Pemeriksaan menyeluruh sistem kelistrikan dan mekanik",
                "Inspeksi keselamatan dan kepatuhan regulasi"
            ],
            'tire_change' => [
                "Penggantian ban depan dan belakang truk {$truck->plate_number}",
                "Rotasi dan penggantian ban yang aus",
                "Penggantian 4 ban dengan merk premium"
            ],
            'filter_change' => [
                "Penggantian filter udara, oli, dan bahan bakar",
                "Pembersihan dan penggantian filter AC",
                "Penggantian semua filter pada interval service"
            ],
            'brake_service' => [
                "Service sistem rem dan penggantian kampas rem",
                "Perbaikan master rem dan sistem hidrolik rem",
                "Penggantian kampas rem dan cakram rem yang aus"
            ],
            'electrical' => [
                "Perbaikan sistem kelistrikan dan penggantian aki",
                "Perbaikan alternator dan sistem pengisian",
                "Penggantian lampu dan perbaikan sistem penerangan"
            ],
            'body_repair' => [
                "Perbaikan body truk yang rusak akibat kecelakaan kecil",
                "Pengecatan ulang bagian kabin truk",
                "Perbaikan bagian bak yang rusak"
            ],
            'other' => [
                "Kalibrasi sistem injeksi bahan bakar",
                "Perbaikan sistem AC kabin",
                "Upgrade sistem navigasi dan GPS",
                "Pemasangan sistem monitoring kendaraan"
            ]
        ];
        
        $typeDescriptions = $descriptions[$type] ?? ["Maintenance untuk truk {$truck->plate_number}"];
        return $typeDescriptions[array_rand($typeDescriptions)];
    }
    
    /**
     * Generate catatan berdasarkan status dan jenis maintenance
     */
    private function generateNotes($status, $type)
    {
        if ($status === 'scheduled') {
            return "Maintenance dijadwalkan, menunggu konfirmasi bengkel";
        } elseif ($status === 'in_progress') {
            return "Sedang dalam proses pengerjaan, estimasi selesai 1-2 hari";
        } else {
            $notes = [
                'oil_change' => [
                    "Oli diganti dengan merk Shell Rimula R4X",
                    "Menggunakan oli synthetic untuk performa lebih baik",
                    "Filter oli juga diganti dengan yang baru"
                ],
                'repair' => [
                    "Suku cadang diganti dengan yang original",
                    "Perlu pemantauan lebih lanjut dalam 1 minggu",
                    "Beberapa komponen perlu penggantian di maintenance berikutnya"
                ],
                'inspection' => [
                    "Semua sistem berfungsi dengan baik",
                    "Ditemukan beberapa masalah kecil yang perlu diperbaiki",
                    "Rekomendasi untuk penggantian beberapa komponen dalam 3 bulan"
                ],
                'tire_change' => [
                    "Ban diganti dengan merk Bridgestone",
                    "Tekanan ban disesuaikan sesuai standar",
                    "Ban lama masih bisa digunakan sebagai cadangan"
                ],
                'filter_change' => [
                    "Semua filter diganti dengan yang baru",
                    "Filter udara dibersihkan dan dipasang kembali",
                    "Rekomendasi untuk penggantian filter lebih rutin"
                ],
                'brake_service' => [
                    "Sistem rem berfungsi dengan baik setelah service",
                    "Kampas rem diganti dengan kualitas premium",
                    "Disarankan untuk pemeriksaan rutin setiap 5000 KM"
                ],
                'electrical' => [
                    "Sistem kelistrikan diperbaiki dan berfungsi normal",
                    "Aki diganti dengan yang baru",
                    "Alternator dibersihkan dan disetel ulang"
                ],
                'body_repair' => [
                    "Body diperbaiki dan dicat ulang sesuai warna asli",
                    "Beberapa bagian diganti dengan panel baru",
                    "Hasil perbaikan sangat baik dan tidak terlihat bekas kerusakan"
                ],
                'other' => [
                    "Maintenance selesai dengan hasil baik",
                    "Performa kendaraan meningkat setelah perbaikan",
                    "Rekomendasi untuk maintenance rutin setiap 3 bulan"
                ]
            ];
            
            $typeNotes = $notes[$type] ?? ["Maintenance selesai dengan baik"];
            return $typeNotes[array_rand($typeNotes)];
        }
    }
}