<?php

namespace App\Http\Controllers;

use App\Models\FuelPurchase;
use App\Models\Expense;
use App\Models\Truck;
use App\Models\Driver;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class FuelPurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $fuelPurchases = FuelPurchase::with('truck')->latest()->paginate(10);
        return view('fuel-purchases.index', compact('fuelPurchases'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $trucks = Truck::where('status', 'active')->get();
        $drivers = Driver::orderBy('name')->get();
        return view('fuel-purchases.create', compact('trucks', 'drivers'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'truck_id' => 'required|exists:trucks,id',
            'purchase_date' => 'required|date',
            'quantity' => 'required|numeric|min:0',
            'price_per_liter' => 'required|numeric|min:0',
            'total_cost' => 'required|numeric|min:0',
            'fuel_type' => 'required|string|max:255',
            'station_name' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'driver_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        
        try {
            // Simpan data pembelian bahan bakar
            $fuelPurchase = FuelPurchase::create($validated);
            
            // Buat catatan pengeluaran otomatis dengan status draft
            $truck = Truck::find($validated['truck_id']);
            $truckInfo = $truck ? $truck->plate_number . ' - ' . $truck->brand . ' ' . $truck->model : 'Truk tidak diketahui';
            
            Expense::create([
                'title' => 'Pembelian Bahan Bakar - ' . $validated['fuel_type'] . ' untuk ' . $truckInfo,
                'description' => 'Pembelian otomatis dari modul bahan bakar. ' . ($validated['notes'] ?? ''),
                'amount' => $validated['total_cost'],
                'expense_date' => $validated['purchase_date'],
                'project_id' => null, // Bisa diisi jika ada project terkait
                'fuel_purchase_id' => $fuelPurchase->id,
                'category' => 'Bahan Bakar',
                'payment_method' => 'Tunai', // Default, bisa disesuaikan
                'receipt_number' => $validated['receipt_number'],
                'status' => 'draft', // Status draft yang memerlukan persetujuan
                'added_by' => Auth::id(),
            ]);
            
            DB::commit();
            
            return redirect()->route('fuel-purchases.index')
                ->with('success', 'Data pembelian bahan bakar berhasil ditambahkan dan menunggu persetujuan pengeluaran.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FuelPurchase $fuelPurchase)
    {
        $validated = $request->validate([
            'truck_id' => 'required|exists:trucks,id',
            'purchase_date' => 'required|date',
            'quantity' => 'required|numeric|min:0',
            'price_per_liter' => 'required|numeric|min:0',
            'total_cost' => 'required|numeric|min:0',
            'fuel_type' => 'required|string|max:255',
            'station_name' => 'nullable|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'driver_name' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        
        try {
            // Update data pembelian bahan bakar
            $fuelPurchase->update($validated);
            
            // Update atau buat catatan pengeluaran
            $truck = Truck::find($validated['truck_id']);
            $truckInfo = $truck ? $truck->plate_number . ' - ' . $truck->brand . ' ' . $truck->model : 'Truk tidak diketahui';
            
            $expense = Expense::where('fuel_purchase_id', $fuelPurchase->id)->first();
            
            $expenseData = [
                'title' => 'Pembelian Bahan Bakar - ' . $validated['fuel_type'] . ' untuk ' . $truckInfo,
                'description' => 'Pembelian otomatis dari modul bahan bakar. ' . ($validated['notes'] ?? ''),
                'amount' => $validated['total_cost'],
                'expense_date' => $validated['purchase_date'],
                'fuel_purchase_id' => $fuelPurchase->id,
                'category' => 'Bahan Bakar',
                'payment_method' => 'Tunai', // Default, bisa disesuaikan
                'receipt_number' => $validated['receipt_number'],
            ];
            
            // Jika pengeluaran sudah disetujui, jangan ubah statusnya
            if ($expense) {
                // Jika status sudah approved, jangan ubah status
                if ($expense->status !== 'approved') {
                    $expenseData['status'] = 'draft';
                }
                $expense->update($expenseData);
            } else {
                $expenseData['status'] = 'draft';
                $expenseData['added_by'] = Auth::id();
                Expense::create($expenseData);
            }
            
            DB::commit();
            
            return redirect()->route('fuel-purchases.index')
                ->with('success', 'Data pembelian bahan bakar berhasil diperbarui.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(FuelPurchase $fuelPurchase)
    {
        return view('fuel-purchases.show', compact('fuelPurchase'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(FuelPurchase $fuelPurchase)
    {
        $trucks = Truck::where('status', 'active')->get();
        $drivers = Driver::orderBy('name')->get();
        return view('fuel-purchases.edit', compact('fuelPurchase', 'trucks', 'drivers'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FuelPurchase $fuelPurchase)
    {
        $fuelPurchase->delete();

        return redirect()->route('fuel-purchases.index')
            ->with('success', 'Data pembelian bahan bakar berhasil dihapus.');
    }
}
