<?php

namespace App\Http\Controllers;

use App\Models\Income;
use App\Models\Project;
use App\Exports\IncomesExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Rap2hpoutre\FastExcel\FastExcel;

class IncomeController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:manage-finances');
    }
    
    /**
     * Display a listing of incomes.
     */
    public function index(Request $request)
    {
        $query = Income::with(['project', 'addedBy']);
        
        // Filter berdasarkan pencarian
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhere('category', 'like', "%{$searchTerm}%");
            });
        }
        
        // Filter berdasarkan tanggal mulai
        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->whereDate('income_date', '>=', $request->start_date);
        }
        
        // Filter berdasarkan tanggal akhir
        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->whereDate('income_date', '<=', $request->end_date);
        }
        
        $incomes = $query->latest()->paginate(10)->withQueryString();
        return view('incomes.index', compact('incomes'));
    }

    /**
     * Show the form for creating a new income.
     */
    public function create()
    {
        $projects = Project::all();
        $categories = ['Pembayaran Proyek', 'Investasi', 'Penjualan Aset', 'Lainnya'];
        $paymentMethods = ['Tunai', 'Transfer Bank', 'Cek', 'Lainnya'];
        
        return view('incomes.create', compact('projects', 'categories', 'paymentMethods'));
    }

    /**
     * Store a newly created income in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'amount' => 'required|numeric|min:0',
            'income_date' => 'required|date',
            'project_id' => 'nullable|exists:projects,id',
            'category' => 'required|string|max:255',
            'payment_method' => 'required|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'payment_proof' => 'nullable|file|mimes:jpeg,png,jpg,pdf|max:2048',
        ]);
        
        $validated['added_by'] = Auth::id();
        $validated['approval_status'] = 'pending';
        
        // Handle file upload
        if ($request->hasFile('payment_proof')) {
            $path = $request->file('payment_proof')->store('payment_proofs', 'public');
            $validated['payment_proof'] = $path;
        }
        
        Income::create($validated);
        
        return redirect()->route('incomes.index')
            ->with('success', 'Pendapatan berhasil ditambahkan dan menunggu approval.');
    }

    /**
     * Display the specified income.
     */
    public function show(Income $income)
    {
        $income->load(['project', 'addedBy', 'approvedBy']);
        return view('incomes.show', compact('income'));
    }

    /**
     * Show the form for editing the specified income.
     */
    public function edit(Income $income)
    {
        $projects = Project::all();
        $categories = ['Pembayaran Proyek', 'Investasi', 'Penjualan Aset', 'Lainnya'];
        $paymentMethods = ['Tunai', 'Transfer Bank', 'Cek', 'Lainnya'];
        
        return view('incomes.edit', compact('income', 'projects', 'categories', 'paymentMethods'));
    }

    /**
     * Update the specified income in storage.
     */
    public function update(Request $request, Income $income)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'amount' => 'required|numeric|min:0',
            'income_date' => 'required|date',
            'project_id' => 'nullable|exists:projects,id',
            'category' => 'required|string|max:255',
            'payment_method' => 'required|string|max:255',
            'receipt_number' => 'nullable|string|max:255',
            'payment_proof' => 'nullable|file|mimes:jpeg,png,jpg,pdf|max:2048',
        ]);
        
        // Reset approval status if important fields are changed
        if ($income->amount != $request->amount || $income->income_date != $request->income_date) {
            $validated['approval_status'] = 'pending';
            $validated['approved_by'] = null;
            $validated['approved_at'] = null;
        }
        
        // Handle file upload
        if ($request->hasFile('payment_proof')) {
            // Delete old file if exists
            if ($income->payment_proof) {
                Storage::disk('public')->delete($income->payment_proof);
            }
            
            $path = $request->file('payment_proof')->store('payment_proofs', 'public');
            $validated['payment_proof'] = $path;
        }
        
        $income->update($validated);
        
        return redirect()->route('incomes.index')
            ->with('success', 'Pendapatan berhasil diperbarui.');
    }

    /**
     * Remove the specified income from storage.
     */
    public function destroy(Income $income)
    {
        // Delete payment proof file if exists
        if ($income->payment_proof) {
            Storage::disk('public')->delete($income->payment_proof);
        }
        
        $income->delete();
        
        return redirect()->route('incomes.index')
            ->with('success', 'Pendapatan berhasil dihapus.');
    }

    /**
     * Approve the specified income.
     */
    public function approve(Income $income)
    {
        $income->update([
            'approval_status' => 'approved',
            'approved_by' => Auth::id(),
            'approved_at' => Carbon::now(),
        ]);
        
        return redirect()->route('incomes.show', $income)
            ->with('success', 'Pendapatan berhasil disetujui.');
    }

    /**
     * Reject the specified income.
     */
    public function reject(Income $income)
    {
        $income->update([
            'approval_status' => 'rejected',
            'approved_by' => Auth::id(),
            'approved_at' => Carbon::now(),
        ]);
        
        return redirect()->route('incomes.show', $income)
            ->with('success', 'Pendapatan berhasil ditolak.');
    }

    /**
     * Export incomes to Excel
     */
    public function export() 
    {
        $exporter = new IncomesExport();
        return (new FastExcel($exporter->collection()))->download('pendapatan.xlsx');
    }
}
