<?php

namespace App\Http\Controllers;

use App\Models\Expense;
use App\Models\Income;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Exports\FinancialReportExport;
use Rap2hpoutre\FastExcel\FastExcel;
use Barryvdh\DomPDF\Facade\Pdf; // Pindahkan ke sini
// Tambahkan model-model berikut
use App\Models\Trip;
use App\Models\Truck;
use App\Models\Driver;
use App\Models\FuelPurchase;
use App\Models\Client; // Add this import statement

class ReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:view-reports');
    }
    
    /**
     * Tampilkan laporan keuangan
     */
    public function financial(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $projectId = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');
        
        // Query untuk expenses
        $expensesQuery = Expense::whereBetween('expense_date', [$startDate, $endDate]);
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $expensesQuery->where('project_id', $projectId);
        }
        
        // Filter berdasarkan kategori jika ada
        if ($category) {
            $expensesQuery->where('category', $category);
        }
        
        // Filter berdasarkan status jika ada
        if ($status) {
            $expensesQuery->where('status', $status);
        }
        
        $expenses = $expensesQuery->get();
        
        // Query untuk incomes
        $incomesQuery = Income::whereBetween('income_date', [$startDate, $endDate]);
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $incomesQuery->where('project_id', $projectId);
        }
        
        // Filter berdasarkan kategori jika ada
        if ($category) {
            $incomesQuery->where('category', $category);
        }
        
        $incomes = $incomesQuery->get();
        
        // Hitung total
        $totalExpenses = $expenses->sum('amount');
        $totalIncomes = $incomes->sum('amount');
        $balance = $totalIncomes - $totalExpenses;
        
        // Dapatkan daftar project untuk filter
        $projects = Project::all();
        
        // Dapatkan daftar kategori unik dari expenses dan incomes
        $expenseCategories = Expense::distinct('category')->pluck('category');
        $incomeCategories = Income::distinct('category')->pluck('category');
        $categories = $expenseCategories->merge($incomeCategories)->unique();
        
        // Dapatkan daftar status untuk filter
        $statuses = ['draft', 'approved', 'rejected'];
        
        return view('reports.financial', compact(
            'expenses', 'incomes', 'totalExpenses', 'totalIncomes', 'balance',
            'startDate', 'endDate', 'projectId', 'category', 'status',
            'projects', 'categories', 'statuses'
        ));
    }
    
    /**
     * Tampilkan laporan pengeluaran
     */
    public function expenses(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $projectId = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');
        
        // Query untuk expenses
        $expensesQuery = Expense::whereBetween('expense_date', [$startDate, $endDate]);
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $expensesQuery->where('project_id', $projectId);
        }
        
        // Filter berdasarkan kategori jika ada
        if ($category) {
            $expensesQuery->where('category', $category);
        }
        
        // Filter berdasarkan status jika ada
        if ($status) {
            $expensesQuery->where('status', $status);
        }
        
        $expenses = $expensesQuery->get();
        $totalExpenses = $expenses->sum('amount');
        
        // Dapatkan daftar project untuk filter
        $projects = Project::all();
        
        // Dapatkan daftar kategori unik dari expenses
        $categories = Expense::distinct('category')->pluck('category');
        
        // Dapatkan daftar status untuk filter
        $statuses = ['draft', 'approved', 'rejected'];
        
        // Hitung pengeluaran per kategori untuk grafik
        $expensesByCategory = $expenses->groupBy('category')
            ->map(function ($items) {
                return $items->sum('amount');
            });
        
        return view('reports.expenses', compact(
            'expenses', 'totalExpenses',
            'startDate', 'endDate', 'projectId', 'category', 'status',
            'projects', 'categories', 'statuses', 'expensesByCategory'
        ));
    }
    
    /**
     * Export financial report to Excel
     */
    public function exportFinancialExcel(Request $request)
    {
        // Ambil data dengan filter yang sama seperti di method financial
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));
        $projectId = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');
        
        // Query untuk pendapatan
        $incomesQuery = Income::whereBetween('income_date', [$startDate, $endDate]);
        if ($projectId) {
            $incomesQuery->where('project_id', $projectId);
        }
        if ($category) {
            $incomesQuery->where('category', $category);
        }
        $incomes = $incomesQuery->get();
        
        // Query untuk pengeluaran
        $expensesQuery = Expense::whereBetween('expense_date', [$startDate, $endDate]);
        if ($projectId) {
            $expensesQuery->where('project_id', $projectId);
        }
        if ($category) {
            $expensesQuery->where('category', $category);
        }
        if ($status) {
            $expensesQuery->where('status', $status);
        }
        $expenses = $expensesQuery->get();
        
        // Buat collection untuk ekspor
        $data = collect();
        
        // Tambahkan data pendapatan
        foreach ($incomes as $income) {
            $data->push([
                'Tipe' => 'Pendapatan',
                'Judul' => $income->title,
                'Jumlah (Rp)' => number_format($income->amount, 2, ',', '.'),
                'Tanggal' => $income->income_date->format('d/m/Y'),
                'Kategori' => $income->category,
                'Proyek' => $income->project ? $income->project->name : 'Tidak Ada',
                'Status' => '-',
                'Ditambahkan Oleh' => $income->addedBy->name
            ]);
        }
        
        // Tambahkan data pengeluaran
        foreach ($expenses as $expense) {
            $data->push([
                'Tipe' => 'Pengeluaran',
                'Judul' => $expense->title,
                'Jumlah (Rp)' => number_format($expense->amount, 2, ',', '.'),
                'Tanggal' => $expense->expense_date->format('d/m/Y'),
                'Kategori' => $expense->category,
                'Proyek' => $expense->project ? $expense->project->name : 'Tidak Ada',
                'Status' => ucfirst($expense->status),
                'Ditambahkan Oleh' => $expense->addedBy->name
            ]);
        }
        
        // Ekspor ke Excel
        return (new FastExcel($data))->download('laporan_keuangan.xlsx');
    }

    /**
     * Export financial report to PDF
     */
    public function exportFinancialPDF(Request $request)
    {
        // Ambil data dengan filter yang sama seperti di method financial
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));
        $projectId = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');
        
        // Query untuk pendapatan
        $incomesQuery = Income::whereBetween('income_date', [$startDate, $endDate]);
        if ($projectId) {
            $incomesQuery->where('project_id', $projectId);
        }
        if ($category) {
            $incomesQuery->where('category', $category);
        }
        $incomes = $incomesQuery->get();
        
        // Query untuk pengeluaran
        $expensesQuery = Expense::whereBetween('expense_date', [$startDate, $endDate]);
        if ($projectId) {
            $expensesQuery->where('project_id', $projectId);
        }
        if ($category) {
            $expensesQuery->where('category', $category);
        }
        if ($status) {
            $expensesQuery->where('status', $status);
        }
        $expenses = $expensesQuery->get();
        
        // Hitung total
        $totalExpenses = $expenses->sum('amount');
        $totalIncomes = $incomes->sum('amount');
        $balance = $totalIncomes - $totalExpenses;
        
        $data = [
            'incomes' => $incomes,
            'expenses' => $expenses,
            'totalIncomes' => $totalIncomes,
            'totalExpenses' => $totalExpenses,
            'balance' => $balance,
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
        
        // Aktifkan baris ini setelah menginstal package dompdf
        return PDF::loadView('reports.financial_pdf', $data)->download('laporan_keuangan.pdf');
    }

    /**
     * Tampilkan laporan perjalanan
     */
    public function trips(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truckId = $request->input('truck_id');
        $driverId = $request->input('driver_id');
        $projectId = $request->input('project_id');
        
        // Query untuk trips
        $tripsQuery = Trip::whereBetween('start_date', [$startDate, $endDate]);
        
        // Filter berdasarkan truck jika ada
        if ($truckId) {
            $tripsQuery->where('truck_id', $truckId);
        }
        
        // Filter berdasarkan driver jika ada
        if ($driverId) {
            $tripsQuery->where('driver_id', $driverId);
        }
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $tripsQuery->where('project_id', $projectId);
        }
        
        $trips = $tripsQuery->get();
        
        // Hitung total jarak dan muatan
        $totalDistance = $trips->sum('distance');
        $totalLoad = $trips->sum('load_weight');
        
        // Dapatkan daftar truck, driver, dan project untuk filter
        $trucks = Truck::all();
        $drivers = Driver::all();
        $projects = Project::all();
        
        // Hitung perjalanan per truck untuk grafik
        $tripsByTruck = $trips->groupBy('truck_id')
            ->map(function ($items, $key) use ($trucks) {
                $truck = $trucks->find($key);
                return [
                    'label' => $truck ? $truck->license_plate : 'Unknown',
                    'count' => $items->count()
                ];
            })
            ->pluck('count', 'label');
        
        // Hitung perjalanan per project untuk grafik
        $tripsByProject = $trips->groupBy('project_id')
            ->map(function ($items, $key) use ($projects) {
                $project = $projects->find($key);
                return [
                    'label' => $project ? $project->name : 'Tidak Ada',
                    'count' => $items->count()
                ];
            })
            ->pluck('count', 'label');
        
        return view('reports.trips', compact(
            'trips', 'totalDistance', 'totalLoad',
            'startDate', 'endDate', 'truckId', 'driverId', 'projectId',
            'trucks', 'drivers', 'projects', 'tripsByTruck', 'tripsByProject'
        ));
    }
    
    /**
     * Tampilkan laporan bahan bakar
     */
    public function fuel(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truckId = $request->input('truck_id');
        $fuelType = $request->input('fuel_type');
        $location = $request->input('location');
        
        // Query untuk fuel purchases
        $fuelQuery = FuelPurchase::whereBetween('purchase_date', [$startDate, $endDate]);
        
        // Filter berdasarkan truck jika ada
        if ($truckId) {
            $fuelQuery->where('truck_id', $truckId);
        }
        
        // Filter berdasarkan jenis bahan bakar jika ada
        if ($fuelType) {
            $fuelQuery->where('fuel_type', $fuelType);
        }
        
        // Filter berdasarkan lokasi jika ada
        if ($location) {
            $fuelQuery->where('location', 'like', "%{$location}%");
        }
        
        $fuelPurchases = $fuelQuery->get();
        
        // Hitung total volume dan biaya
        $totalVolume = $fuelPurchases->sum('volume');
        $totalCost = $fuelPurchases->sum('total_cost');
        
        // Dapatkan daftar truck untuk filter
        $trucks = Truck::all();
        
        // Dapatkan daftar jenis bahan bakar unik untuk filter
        $fuelTypes = FuelPurchase::distinct('fuel_type')->pluck('fuel_type');
        
        // Hitung konsumsi bahan bakar per truck untuk grafik
        $fuelByTruck = $fuelPurchases->groupBy('truck_id')
            ->map(function ($items, $key) use ($trucks) {
                $truck = $trucks->find($key);
                return [
                    'label' => $truck ? $truck->license_plate : 'Unknown',
                    'volume' => $items->sum('volume')
                ];
            })
            ->pluck('volume', 'label');
        
        // Hitung biaya bahan bakar per bulan untuk grafik
        $fuelByMonth = $fuelPurchases
            ->groupBy(function ($item) {
                return $item->purchase_date->format('M Y');
            })
            ->map(function ($items) {
                return $items->sum('total_cost');
            });
        
        return view('reports.fuel', compact(
            'fuelPurchases', 'totalVolume', 'totalCost',
            'startDate', 'endDate', 'truckId', 'fuelType', 'location',
            'trucks', 'fuelTypes', 'fuelByTruck', 'fuelByMonth'
        ));
    }

    /**
     * Tampilkan laporan proyek
     */
    public function projects(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $clientId = $request->input('client_id');
        
        // Query untuk projects
        $projectsQuery = Project::query();
        
        // Filter berdasarkan client jika ada
        if ($clientId) {
            $projectsQuery->where('client_id', $clientId);
        }
        
        $projects = $projectsQuery->get();
        
        // Hitung total trips dan expenses per proyek
        $projectStats = collect();
        
        foreach ($projects as $project) {
            $trips = Trip::where('project_id', $project->id)
                ->whereBetween('start_date', [$startDate, $endDate])
                ->get();
                
            $expenses = Expense::where('project_id', $project->id)
                ->whereBetween('expense_date', [$startDate, $endDate])
                ->get();
                
            $incomes = Income::where('project_id', $project->id)
                ->whereBetween('income_date', [$startDate, $endDate])
                ->get();
            
            $projectStats->push([
                'project' => $project,
                'trips_count' => $trips->count(),
                'total_distance' => $trips->sum('distance'),
                'total_load' => $trips->sum('load_weight'),
                'expenses_total' => $expenses->sum('amount'),
                'incomes_total' => $incomes->sum('amount'),
                'profit' => $incomes->sum('amount') - $expenses->sum('amount')
            ]);
        }
        
        // Dapatkan daftar client untuk filter
        $clients = Client::all();
        
        return view('reports.projects', compact(
            'projectStats', 'startDate', 'endDate', 'clientId', 'clients'
        ));
    }

    /**
     * Export projects report to Excel
     */
    public function exportProjectsExcel(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $clientId = $request->input('client_id');
        
        // Query untuk projects
        $projectsQuery = Project::query();
        
        // Filter berdasarkan client jika ada
        if ($clientId) {
            $projectsQuery->where('client_id', $clientId);
        }
        
        $projects = $projectsQuery->get();
        
        // Buat collection untuk ekspor
        $data = collect();
        
        foreach ($projects as $project) {
            $trips = Trip::where('project_id', $project->id)
                ->whereBetween('start_date', [$startDate, $endDate])
                ->get();
                
            $expenses = Expense::where('project_id', $project->id)
                ->whereBetween('expense_date', [$startDate, $endDate])
                ->get();
                
            $incomes = Income::where('project_id', $project->id)
                ->whereBetween('income_date', [$startDate, $endDate])
                ->get();
            
            $data->push([
                'ID' => $project->id,
                'Nama Proyek' => $project->name,
                'Klien' => $project->client ? $project->client->name : 'Tidak Ada',
                'Lokasi' => $project->location,
                'Tanggal Mulai' => $project->start_date ? $project->start_date->format('d/m/Y') : '-',
                'Tanggal Selesai' => $project->end_date ? $project->end_date->format('d/m/Y') : '-',
                'Jumlah Perjalanan' => $trips->count(),
                'Total Jarak (km)' => number_format($trips->sum('distance'), 2, ',', '.'),
                'Total Muatan (ton)' => number_format($trips->sum('load_weight'), 2, ',', '.'),
                'Total Pengeluaran (Rp)' => number_format($expenses->sum('amount'), 2, ',', '.'),
                'Total Pendapatan (Rp)' => number_format($incomes->sum('amount'), 2, ',', '.'),
                'Profit (Rp)' => number_format($incomes->sum('amount') - $expenses->sum('amount'), 2, ',', '.')
            ]);
        }
        
        // Ekspor ke Excel
        return (new FastExcel($data))->download('laporan_proyek.xlsx');
    }

    /**
     * Export projects report to PDF
     */
    public function exportProjectsPDF(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $clientId = $request->input('client_id');
        
        // Query untuk projects
        $projectsQuery = Project::query();
        
        // Filter berdasarkan client jika ada
        if ($clientId) {
            $projectsQuery->where('client_id', $clientId);
        }
        
        $projects = $projectsQuery->get();
        
        // Hitung total trips dan expenses per proyek
        $projectStats = collect();
        
        foreach ($projects as $project) {
            $trips = Trip::where('project_id', $project->id)
                ->whereBetween('start_date', [$startDate, $endDate])
                ->get();
                
            $expenses = Expense::where('project_id', $project->id)
                ->whereBetween('expense_date', [$startDate, $endDate])
                ->get();
                
            $incomes = Income::where('project_id', $project->id)
                ->whereBetween('income_date', [$startDate, $endDate])
                ->get();
            
            $projectStats->push([
                'project' => $project,
                'trips_count' => $trips->count(),
                'total_distance' => $trips->sum('distance'),
                'total_load' => $trips->sum('load_weight'),
                'expenses_total' => $expenses->sum('amount'),
                'incomes_total' => $incomes->sum('amount'),
                'profit' => $incomes->sum('amount') - $expenses->sum('amount')
            ]);
        }
        
        $data = [
            'projectStats' => $projectStats,
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
        
        return PDF::loadView('reports.projects_pdf', $data)->download('laporan_proyek.pdf');
    }

    /**
     * Export trips report to Excel
     */
    public function exportTripsExcel(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truckId = $request->input('truck_id');
        $driverId = $request->input('driver_id');
        $projectId = $request->input('project_id');
        
        // Query untuk trips
        $tripsQuery = Trip::whereBetween('start_date', [$startDate, $endDate]);
        
        // Filter berdasarkan truck jika ada
        if ($truckId) {
            $tripsQuery->where('truck_id', $truckId);
        }
        
        // Filter berdasarkan driver jika ada
        if ($driverId) {
            $tripsQuery->where('driver_id', $driverId);
        }
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $tripsQuery->where('project_id', $projectId);
        }
        
        $trips = $tripsQuery->with(['truck', 'driver', 'project'])->get();
        
        // Buat collection untuk ekspor
        $data = collect();
        
        foreach ($trips as $trip) {
            $data->push([
                'ID' => $trip->id,
                'Tanggal Mulai' => $trip->start_date->format('d/m/Y'),
                'Tanggal Selesai' => $trip->end_date ? $trip->end_date->format('d/m/Y') : '-',
                'Truk' => $trip->truck ? $trip->truck->license_plate : '-',
                'Sopir' => $trip->driver ? $trip->driver->name : '-',
                'Proyek' => $trip->project ? $trip->project->name : '-',
                'Jarak (km)' => number_format($trip->distance, 2, ',', '.'),
                'Muatan (ton)' => number_format($trip->load_weight, 2, ',', '.'),
                'Lokasi Asal' => $trip->origin_location,
                'Lokasi Tujuan' => $trip->destination_location,
                'Catatan' => $trip->notes
            ]);
        }
        
        // Ekspor ke Excel
        return (new FastExcel($data))->download('laporan_perjalanan.xlsx');
    }

    /**
     * Export trips report to PDF
     */
    public function exportTripsPDF(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truckId = $request->input('truck_id');
        $driverId = $request->input('driver_id');
        $projectId = $request->input('project_id');
        
        // Query untuk trips
        $tripsQuery = Trip::whereBetween('start_date', [$startDate, $endDate]);
        
        // Filter berdasarkan truck jika ada
        if ($truckId) {
            $tripsQuery->where('truck_id', $truckId);
        }
        
        // Filter berdasarkan driver jika ada
        if ($driverId) {
            $tripsQuery->where('driver_id', $driverId);
        }
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $tripsQuery->where('project_id', $projectId);
        }
        
        $trips = $tripsQuery->with(['truck', 'driver', 'project'])->get();
        
        // Hitung total jarak dan muatan
        $totalDistance = $trips->sum('distance');
        $totalLoad = $trips->sum('load_weight');
        
        $data = [
            'trips' => $trips,
            'totalDistance' => $totalDistance,
            'totalLoad' => $totalLoad,
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
        
        return PDF::loadView('reports.trips_pdf', $data)->download('laporan_perjalanan.pdf');
    }

    /**
     * Export fuel report to Excel
     */
    public function exportFuelExcel(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truckId = $request->input('truck_id');
        $fuelType = $request->input('fuel_type');
        $location = $request->input('location');
        
        // Query untuk fuel purchases
        $fuelQuery = FuelPurchase::whereBetween('purchase_date', [$startDate, $endDate]);
        
        // Filter berdasarkan truck jika ada
        if ($truckId) {
            $fuelQuery->where('truck_id', $truckId);
        }
        
        // Filter berdasarkan jenis bahan bakar jika ada
        if ($fuelType) {
            $fuelQuery->where('fuel_type', $fuelType);
        }
        
        // Filter berdasarkan lokasi jika ada
        if ($location) {
            $fuelQuery->where('location', 'like', "%{$location}%");
        }
        
        $fuelPurchases = $fuelQuery->with(['truck'])->get();
        
        // Buat collection untuk ekspor
        $data = collect();
        
        foreach ($fuelPurchases as $fuel) {
            $data->push([
                'ID' => $fuel->id,
                'Tanggal Pembelian' => $fuel->purchase_date->format('d/m/Y'),
                'Truk' => $fuel->truck ? $fuel->truck->license_plate : '-',
                'Jenis Bahan Bakar' => $fuel->fuel_type,
                'Volume (liter)' => number_format($fuel->volume, 2, ',', '.'),
                'Harga per Liter (Rp)' => number_format($fuel->price_per_liter, 2, ',', '.'),
                'Total Biaya (Rp)' => number_format($fuel->total_cost, 2, ',', '.'),
                'Lokasi' => $fuel->location,
                'Catatan' => $fuel->notes
            ]);
        }
        
        // Ekspor ke Excel
        return (new FastExcel($data))->download('laporan_bahan_bakar.xlsx');
    }

    /**
     * Export fuel report to PDF
     */
    public function exportFuelPDF(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $truckId = $request->input('truck_id');
        $fuelType = $request->input('fuel_type');
        $location = $request->input('location');
        
        // Query untuk fuel purchases
        $fuelQuery = FuelPurchase::whereBetween('purchase_date', [$startDate, $endDate]);
        
        // Filter berdasarkan truck jika ada
        if ($truckId) {
            $fuelQuery->where('truck_id', $truckId);
        }
        
        // Filter berdasarkan jenis bahan bakar jika ada
        if ($fuelType) {
            $fuelQuery->where('fuel_type', $fuelType);
        }
        
        // Filter berdasarkan lokasi jika ada
        if ($location) {
            $fuelQuery->where('location', 'like', "%{$location}%");
        }
        
        $fuelPurchases = $fuelQuery->with(['truck'])->get();
        
        // Hitung total volume dan biaya
        $totalVolume = $fuelPurchases->sum('volume');
        $totalCost = $fuelPurchases->sum('total_cost');
        
        $data = [
            'fuelPurchases' => $fuelPurchases,
            'totalVolume' => $totalVolume,
            'totalCost' => $totalCost,
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
        
        return PDF::loadView('reports.fuel_pdf', $data)->download('laporan_bahan_bakar.pdf');
    }

    /**
     * Export expenses report to Excel
     */
    public function exportExpensesExcel(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $projectId = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');
        
        // Query untuk expenses
        $expensesQuery = Expense::whereBetween('expense_date', [$startDate, $endDate]);
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $expensesQuery->where('project_id', $projectId);
        }
        
        // Filter berdasarkan kategori jika ada
        if ($category) {
            $expensesQuery->where('category', $category);
        }
        
        // Filter berdasarkan status jika ada
        if ($status) {
            $expensesQuery->where('status', $status);
        }
        
        $expenses = $expensesQuery->with(['project', 'addedBy'])->get();
        
        // Buat collection untuk ekspor
        $data = collect();
        
        foreach ($expenses as $expense) {
            $data->push([
                'ID' => $expense->id,
                'Judul' => $expense->title,
                'Deskripsi' => $expense->description,
                'Jumlah (Rp)' => number_format($expense->amount, 2, ',', '.'),
                'Tanggal Pengeluaran' => $expense->expense_date->format('d/m/Y'),
                'Proyek' => $expense->project ? $expense->project->name : 'Tidak Ada',
                'Kategori' => $expense->category,
                'Status' => ucfirst($expense->status),
                'Metode Pembayaran' => $expense->payment_method,
                'Nomor Kwitansi' => $expense->receipt_number ?? '-',
                'Ditambahkan Oleh' => $expense->addedBy->name
            ]);
        }
        
        // Ekspor ke Excel
        return (new FastExcel($data))->download('laporan_pengeluaran.xlsx');
    }

    /**
     * Export expenses report to PDF
     */
    public function exportExpensesPDF(Request $request)
    {
        // Default filter: bulan ini
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $projectId = $request->input('project_id');
        $category = $request->input('category');
        $status = $request->input('status');
        
        // Query untuk expenses
        $expensesQuery = Expense::whereBetween('expense_date', [$startDate, $endDate]);
        
        // Filter berdasarkan project jika ada
        if ($projectId) {
            $expensesQuery->where('project_id', $projectId);
        }
        
        // Filter berdasarkan kategori jika ada
        if ($category) {
            $expensesQuery->where('category', $category);
        }
        
        // Filter berdasarkan status jika ada
        if ($status) {
            $expensesQuery->where('status', $status);
        }
        
        $expenses = $expensesQuery->with(['project', 'addedBy'])->get();
        $totalExpenses = $expenses->sum('amount');
        
        $data = [
            'expenses' => $expenses,
            'totalExpenses' => $totalExpenses,
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
        
        return PDF::loadView('reports.expenses_pdf', $data)->download('laporan_pengeluaran.pdf');
 
    }
}