<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;

class SettingController extends Controller
{
    /**
     * Display the settings page.
     */
    public function index(Request $request)
    {
        $group = $request->query('group', 'general');
        $settings = Setting::where('group', $group)->get();
        
        return view('settings.index', [
            'settings' => $settings,
            'currentGroup' => $group,
            'groups' => [  // You're using 'groups' here
                'general' => ['name' => 'Umum', 'icon' => 'fas fa-cog'],
                'email' => ['name' => 'Email', 'icon' => 'fas fa-envelope'],
                'datetime' => ['name' => 'Tanggal & Waktu', 'icon' => 'fas fa-calendar-alt'],
                'appearance' => ['name' => 'Tampilan', 'icon' => 'fas fa-palette'],
                'notification' => ['name' => 'Notifikasi', 'icon' => 'fas fa-bell']
            ]
        ]);
    }

    /**
     * Update the settings.
     */
    public function update(Request $request)
    {
        $group = $request->input('group', 'general');
        $inputs = $request->except(['_token', 'group']);
        
        foreach ($inputs as $key => $value) {
            $setting = Setting::where('key', $key)->first();
            
            if ($setting) {
                $setting->update(['value' => $value]);
            }
        }
        
        return redirect()->route('settings.index', ['group' => $group])
            ->with('success', 'Pengaturan berhasil disimpan.');
    }

    /**
     * Reset settings to default values.
     */
    public function reset(Request $request)
    {
        $group = $request->input('group', 'general');
        
        // Delete all settings in the group
        Setting::where('group', $group)->delete();
        
        // Re-run the seeder for this group
        $seeder = new \Database\Seeders\SettingSeeder();
        $seeder->run();
        
        return redirect()->route('settings.index', ['group' => $group])
            ->with('success', 'Pengaturan berhasil direset ke nilai default.');
    }

    /**
     * Export settings
     */
    public function export()
    {
        $settings = Setting::all()->groupBy('group');
        return response()->json($settings);
    }

    /**
     * Import settings
     */
    public function import(Request $request)
    {
        try {
            $settingsData = json_decode($request->input('settings'), true);
            
            if (!$settingsData) {
                return response()->json(['success' => false, 'message' => 'Data pengaturan tidak valid.']);
            }
            
            foreach ($settingsData as $group => $settings) {
                foreach ($settings as $setting) {
                    Setting::set(
                        $setting['key'],
                        $setting['value'],
                        $setting['group'],
                        $setting['type'],
                        $setting['description'],
                        $setting['help_text'] ?? null
                    );
                }
            }
            
            return response()->json(['success' => true, 'message' => 'Pengaturan berhasil diimpor.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Gagal mengimpor pengaturan: ' . $e->getMessage()]);
        }
    }

    /**
     * Test email configuration
     */
    public function testEmail(Request $request)
    {
        try {
            $recipientEmail = $request->input('test_email');
            
            // Validasi email
            $validator = Validator::make(['email' => $recipientEmail], [
                'email' => 'required|email'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Alamat email tidak valid.'
                ]);
            }
            
            // Ambil konfigurasi email dari database
            $mailDriver = Setting::get('mail_driver');
            
            // Konfigurasi mailer
            $config = [
                'transport' => $mailDriver, // Gunakan 'transport' bukan 'driver'
                'host' => Setting::get('mail_host'),
                'port' => Setting::get('mail_port'),
                'username' => Setting::get('mail_username'),
                'password' => Setting::get('mail_password'),
                'encryption' => Setting::get('mail_encryption'),
                'timeout' => null,
            ];
            
            $from = [
                'address' => Setting::get('mail_from_address'),
                'name' => Setting::get('mail_from_name'),
            ];
            
            // Konfigurasi mail untuk test
            config(['mail.default' => $mailDriver]); // Tambahkan ini
            config(['mail.mailers.' . $mailDriver => $config]);
            config(['mail.from' => $from]);
            
            // Kirim email test
            Mail::raw('Ini adalah email test dari sistem OBM. Jika Anda menerima email ini, berarti konfigurasi email Anda berhasil.', function($message) use ($recipientEmail) {
                $message->to($recipientEmail)
                        ->subject('Test Email dari OBM');
            });
            
            return response()->json([
                'success' => true, 
                'message' => 'Email test berhasil dikirim ke ' . $recipientEmail
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false, 
                'message' => 'Gagal mengirim email test: ' . $e->getMessage()
            ]);
        }
    }
}