<?php

namespace App\Http\Controllers;

use App\Models\Trip;
use App\Models\Driver;
use App\Models\Truck;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TripController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $trips = Trip::with(['driver', 'truck', 'project'])->latest()->paginate(10);
        return view('trips.index', compact('trips'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $drivers = Driver::all();
        $trucks = Truck::all();
        $projects = Project::all();
        $statuses = ['ongoing' => 'Sedang Berlangsung', 'completed' => 'Selesai', 'cancelled' => 'Dibatalkan'];
        
        return view('trips.create', compact('drivers', 'trucks', 'projects', 'statuses'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'driver_id' => 'required|exists:drivers,id',
            'truck_id' => 'required|exists:trucks,id',
            'project_id' => 'nullable|exists:projects,id',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'start_odometer' => 'required|numeric|min:0',
            'end_odometer' => 'nullable|numeric|min:0|gt:start_odometer',
            'distance' => 'nullable|numeric|min:0',
            'fuel_consumption' => 'nullable|numeric|min:0',
            'status' => 'required|in:ongoing,completed,cancelled',
            'notes' => 'nullable|string',
        ]);
        
        // Hitung jarak jika end_odometer diisi
        if ($request->filled('end_odometer') && !$request->filled('distance')) {
            $validated['distance'] = $request->end_odometer - $request->start_odometer;
        }
        
        Trip::create($validated);
        
        return redirect()->route('trips.index')
            ->with('success', 'Perjalanan berhasil ditambahkan!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Trip $trip)
    {
        $trip->load(['driver', 'truck', 'project']);
        return view('trips.show', compact('trip'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Trip $trip)
    {
        $drivers = Driver::all();
        $trucks = Truck::all();
        $projects = Project::all();
        $statuses = ['ongoing' => 'Sedang Berlangsung', 'completed' => 'Selesai', 'cancelled' => 'Dibatalkan'];
        
        return view('trips.edit', compact('trip', 'drivers', 'trucks', 'projects', 'statuses'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Trip $trip)
    {
        $validated = $request->validate([
            'driver_id' => 'required|exists:drivers,id',
            'truck_id' => 'required|exists:trucks,id',
            'project_id' => 'nullable|exists:projects,id',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'start_odometer' => 'required|numeric|min:0',
            'end_odometer' => 'nullable|numeric|min:0|gt:start_odometer',
            'distance' => 'nullable|numeric|min:0',
            'fuel_consumption' => 'nullable|numeric|min:0',
            'status' => 'required|in:ongoing,completed,cancelled',
            'notes' => 'nullable|string',
        ]);
        
        // Hitung jarak jika end_odometer diisi
        if ($request->filled('end_odometer') && !$request->filled('distance')) {
            $validated['distance'] = $request->end_odometer - $request->start_odometer;
        }
        
        $trip->update($validated);
        
        return redirect()->route('trips.index')
            ->with('success', 'Perjalanan berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Trip $trip)
    {
        $trip->delete();
        
        return redirect()->route('trips.index')
            ->with('success', 'Perjalanan berhasil dihapus!');
    }
}
