<?php

namespace App\Http\Controllers;

use App\Models\Truck;
use Illuminate\Http\Request;

class TruckController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $trucks = Truck::latest()->paginate(10);
        return view('trucks.index', compact('trucks'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('trucks.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'plate_number' => 'required|string|max:255|unique:trucks',
            'brand' => 'required|string|max:255',
            'model' => 'required|string|max:255',
            'year' => 'required|integer|min:1900|max:' . (date('Y') + 1),
            'capacity' => 'required|string|max:255',
            'fuel_type' => 'required|string|max:255',
            'status' => 'required|string|in:active,maintenance,inactive',
            'purchase_date' => 'required|date',
            'last_service_date' => 'nullable|date',
            'next_service_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        Truck::create($validated);

        return redirect()->route('trucks.index')
            ->with('success', 'Truck berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Truck $truck)
    {
        return view('trucks.show', compact('truck'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Truck $truck)
    {
        return view('trucks.edit', compact('truck'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Truck $truck)
    {
        $validated = $request->validate([
            'plate_number' => 'required|string|max:255|unique:trucks,plate_number,' . $truck->id,
            'brand' => 'required|string|max:255',
            'model' => 'required|string|max:255',
            'year' => 'required|integer|min:1900|max:' . (date('Y') + 1),
            'capacity' => 'required|string|max:255',
            'fuel_type' => 'required|string|max:255',
            'status' => 'required|string|in:active,maintenance,inactive',
            'purchase_date' => 'required|date',
            'last_service_date' => 'nullable|date',
            'next_service_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        $truck->update($validated);

        return redirect()->route('trucks.index')
            ->with('success', 'Data truck berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Truck $truck)
    {
        $truck->delete();

        return redirect()->route('trucks.index')
            ->with('success', 'Truck berhasil dihapus.');
    }
}
