<?php

namespace Database\Seeders;

use App\Models\Income;
use App\Models\Project;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;

class IncomeSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Dapatkan semua ID project yang tersedia
        $projectIds = Project::pluck('id')->toArray();
        
        // Dapatkan ID admin untuk added_by dan approved_by
        $adminId = User::where('email', 'admin@obm.com')->first()->id ?? 1;
        
        // Kategori pendapatan yang tersedia
        $categories = ['Pembayaran Proyek', 'Investasi', 'Penjualan Aset', 'Lainnya'];
        
        // Metode pembayaran yang tersedia
        $paymentMethods = ['Tunai', 'Transfer Bank', 'Cek', 'Lainnya'];
        
        // Status persetujuan
        $approvalStatuses = ['pending', 'approved', 'rejected'];
        
        // Buat 30 data pendapatan acak
        for ($i = 0; $i < 30; $i++) {
            // Pilih project secara acak (atau null untuk pendapatan tanpa project)
            $projectId = rand(0, 1) ? $projectIds[array_rand($projectIds)] : null;
            
            // Hitung tanggal pendapatan (dalam 6 bulan terakhir)
            $incomeDate = Carbon::now()->subDays(rand(1, 180));
            
            // Tentukan status persetujuan
            $approvalStatus = $approvalStatuses[array_rand($approvalStatuses)];
            
            // Data dasar pendapatan
            $incomeData = [
                'title' => 'Pendapatan ' . fake()->words(3, true),
                'description' => fake()->optional(0.7)->paragraph(),
                'amount' => rand(1000000, 50000000) + (rand(0, 99) / 100), // Rp 1.000.000 - Rp 50.000.000
                'income_date' => $incomeDate,
                'project_id' => $projectId,
                'category' => $categories[array_rand($categories)],
                'payment_method' => $paymentMethods[array_rand($paymentMethods)],
                'receipt_number' => fake()->optional(0.8)->bothify('REC-####-????'),
                'approval_status' => $approvalStatus,
                'added_by' => $adminId,
            ];
            
            // Jika status approved, tambahkan data approval
            if ($approvalStatus === 'approved') {
                $incomeData['approved_by'] = $adminId;
                $incomeData['approved_at'] = Carbon::parse($incomeDate)->addDays(rand(1, 5));
                
                // 50% kemungkinan memiliki bukti pembayaran
                if (rand(0, 1)) {
                    $incomeData['payment_proof'] = 'payment_proofs/dummy-receipt-' . rand(1, 5) . '.jpg';
                }
            }
            
            // Buat data pendapatan
            Income::create($incomeData);
        }
        
        // Tambahkan beberapa data spesifik dengan status approved
        Income::create([
            'title' => 'Pembayaran Tahap 1 Proyek Jalan',
            'description' => 'Pembayaran pertama untuk proyek pembangunan jalan',
            'amount' => 25000000.00,
            'income_date' => Carbon::now()->subDays(10),
            'project_id' => $projectIds[0],
            'category' => 'Pembayaran Proyek',
            'payment_method' => 'Transfer Bank',
            'receipt_number' => 'REC-2023-0123',
            'approval_status' => 'approved',
            'approved_by' => $adminId,
            'approved_at' => Carbon::now()->subDays(8),
            'payment_proof' => 'payment_proofs/dummy-receipt-bank.jpg',
            'added_by' => $adminId,
        ]);
        
        Income::create([
            'title' => 'Penjualan Truk Bekas',
            'description' => 'Penjualan truk bekas yang sudah tidak digunakan',
            'amount' => 150000000.00,
            'income_date' => Carbon::now()->subDays(15),
            'project_id' => null,
            'category' => 'Penjualan Aset',
            'payment_method' => 'Tunai',
            'receipt_number' => 'REC-2023-0456',
            'approval_status' => 'approved',
            'approved_by' => $adminId,
            'approved_at' => Carbon::now()->subDays(14),
            'payment_proof' => 'payment_proofs/dummy-receipt-cash.jpg',
            'added_by' => $adminId,
        ]);
        
        Income::create([
            'title' => 'Investasi Dari Partner',
            'description' => 'Investasi modal dari partner bisnis',
            'amount' => 100000000.00,
            'income_date' => Carbon::now()->subDays(5),
            'project_id' => null,
            'category' => 'Investasi',
            'payment_method' => 'Transfer Bank',
            'receipt_number' => 'REC-2023-0789',
            'approval_status' => 'pending',
            'added_by' => $adminId,
        ]);
        
        // Tambahkan contoh pendapatan yang ditolak
        Income::create([
            'title' => 'Pembayaran Tidak Valid',
            'description' => 'Pembayaran yang ditolak karena informasi tidak lengkap',
            'amount' => 5000000.00,
            'income_date' => Carbon::now()->subDays(20),
            'project_id' => $projectIds[array_rand($projectIds)],
            'category' => 'Pembayaran Proyek',
            'payment_method' => 'Cek',
            'receipt_number' => 'REC-2023-9999',
            'approval_status' => 'rejected',
            'approved_by' => $adminId,
            'approved_at' => Carbon::now()->subDays(18),
            'added_by' => $adminId,
        ]);
    }
}